package de.upb.pga3.panda2.client.core;

import java.io.File;

import de.upb.pga3.panda2.client.core.datastructures.ManifestInfo;
import de.upb.pga3.panda2.client.core.datastructures.UIMessage;
import de.upb.pga3.panda2.client.core.datastructures.UIMessageType;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.core.services.XMLParser;
import de.upb.pga3.panda2.utilities.Constants;

/**
 * Class for identifying the manifest and fingerprint information of the
 * application and also for comparing two applications.
 *
 * @author Sriram
 *
 */
public class AppIdentifier {

	/*
	 * Variable for accessing the manifest information of the first application.
	 */
	private ManifestInfo manifestInfo1;
	/*
	 * Variable for accessing the manifest information of the second
	 * application.
	 */
	private ManifestInfo manifestInfo2;
	// Variable for persisting the fingerprint value of first application
	private String app1FingerPrint;
	// Variable for persisting the fingerprint value of second application
	private String app2FingerPrint;
	// Variable for XMLParser
	private XMLParser a3XMLParser;

	public AppIdentifier() {
		this.a3XMLParser = CoreServices.getXMLParserInstance();
	}

	/**
	 * Get the instance of the ManifestInfo class (information of first
	 * application).
	 *
	 * @return manifestInfo1, instance of the ManifestInfo class.
	 */
	public ManifestInfo getManifestInfo1() {
		return this.manifestInfo1;
	}

	/**
	 * Set the instance of the ManifestInfo class (information of first
	 * application).
	 *
	 * @param manifestInfo1
	 *            Instance of ManifestInfo class.
	 */
	public void setManifestInfo1(final ManifestInfo manifestInfo1) {
		this.manifestInfo1 = manifestInfo1;
	}

	/**
	 * Get the instance of the ManifestInfo class (information of second
	 * application).
	 *
	 * @return manifestInfo2, instance of the ManifestInfo class.
	 */
	public ManifestInfo getManifestInfo2() {
		return this.manifestInfo2;
	}

	/**
	 * Set the instance of the ManifestInfo class (information of first
	 * application).
	 *
	 * @param manifestInfo2
	 *            Instance of ManifestInfo class.
	 */
	public void setManifestInfo2(final ManifestInfo manifestInfo2) {
		this.manifestInfo2 = manifestInfo2;
	}

	/**
	 * Get the fingerprint value of the first application.
	 *
	 * @return app1FingerPrint, String value.
	 */
	public String getApp1FingerPrint() {
		return this.app1FingerPrint;
	}

	/**
	 * Set the fingerprint value of the first application.
	 *
	 * @param app1FingerPrint
	 *            The fingerprint value as String.
	 */
	public void setApp1FingerPrint(final String app1FingerPrint) {
		this.app1FingerPrint = app1FingerPrint;
	}

	/**
	 * Get the fingerprint value of the second application.
	 *
	 * @return app2FingerPrint, String value.
	 */
	public String getApp2FingerPrint() {
		return this.app2FingerPrint;
	}

	/**
	 * Set the fingerprint value of the second application.
	 *
	 * @param app1FingerPrint
	 *            The fingerprint value as String.
	 */
	public void setApp2FingerPrint(final String app2FingerPrint) {
		this.app2FingerPrint = app2FingerPrint;
	}

	/**
	 * Method for comparing the information of two applications.
	 *
	 * @param apk
	 *            The input apk as a File instance. Considered as first
	 *            application in comparison scenario.
	 * @param prevAnalResult
	 *            The previous analysis result as AnalysisResult instance.
	 *            Considered as second application in comparison scenario.
	 * @return UIMessage
	 */
	public UIMessage compareAppWithPreviousResult(final File apk, final File prevAnalResult) {
		if (prevAnalResult.toString()
				.substring(prevAnalResult.toString().length() - 4, prevAnalResult.toString().length())
				.equals(Constants.SAVED_RESULT_EXTENSION)) {
			final AnalysisResult loadedResult = ResultLoader.loadPreviousAnalysisResult(prevAnalResult.toString());
			return compareAppWithPreviousResult(apk, loadedResult);
		} else {
			return compareAppWithApp(apk, prevAnalResult);
		}
	}

	/**
	 * Method for comparing the apk manifest information with previous analysis
	 * result.
	 *
	 * @param apk
	 *            The input apk as a File instance. Considered as first
	 *            application in comparison scenario.
	 * @param prevAnalResult
	 *            The previous analysis result as AnalysisResult instance (.pa2
	 *            file). Considered as second application in comparison
	 *            scenario.
	 * @return UIMessage
	 */
	public UIMessage compareAppWithPreviousResult(final File apk, final AnalysisResult prevAnalResult) {
		this.manifestInfo1 = getManifestInformation(apk);
		this.app1FingerPrint = getFingerPrint(apk);
		this.manifestInfo2 = getVersionInfoOfPreviousAnalysisResult(prevAnalResult);
		this.app2FingerPrint = getFingerPrintOfPreviousAnalysisResult(prevAnalResult);
		return compareManifestAndFingerPrintInformation();
	}

	/**
	 * Method for comparing the apk manifest information with another apk.
	 *
	 * @param apk
	 *            The input apk as a File instance. Considered as first
	 *            application in comparison scenario.
	 * @param apk
	 *            The input apk as a File instance. Considered as second
	 *            application in comparison scenario.
	 * @return UIMessage
	 */
	public UIMessage compareAppWithApp(final File apk1, final File apk2) {
		this.manifestInfo1 = getManifestInformation(apk1);
		this.app1FingerPrint = getFingerPrint(apk1);
		this.manifestInfo2 = getManifestInformation(apk2);
		this.app2FingerPrint = getFingerPrint(apk2);
		return compareManifestAndFingerPrintInformation();
	}

	/**
	 * Method for getting the manifest information of the first application
	 * using the apk file.
	 *
	 * @param apk
	 *            Instance of File class.
	 * @return manifestInfo, the instance of the {@code ManifestInfo} class.
	 */
	private ManifestInfo getManifestInformation(final File apk) {
		this.a3XMLParser.fetchData(apk.toString());
		return this.a3XMLParser.getManifestInformation();
	}

	/**
	 * Method for getting the fingerprint information of the first application
	 * using the apk file.
	 *
	 * @param apk
	 *            Instance of File class.
	 * @return the fingerprint value as String.
	 */
	private String getFingerPrint(final File apk) {
		this.a3XMLParser.fetchData(apk.toString());
		return this.a3XMLParser.getFingerprint();
	}

	/**
	 * Method for getting the manifest information of the second application
	 * using previous analysis result.
	 *
	 * @param analysisResult
	 *            Instance of AnalysisResult class.
	 * @return manifestInfo, the instance of the {@code ManifestInfo} class.
	 */
	private static ManifestInfo getVersionInfoOfPreviousAnalysisResult(final AnalysisResult analysisResult) {
		return analysisResult.getAppManifestInfo();
	}

	/**
	 * Method for getting the fingerprint information of the first application
	 * using the apk file.
	 *
	 * @param analysisResult
	 *            Instance of File class.
	 * @return the fingerprint value as String.
	 */
	private static String getFingerPrintOfPreviousAnalysisResult(final AnalysisResult analysisResult) {
		return analysisResult.getAppFingerprint();
	}

	/**
	 * Method for comparing the mainfest and fingerprint information of both the
	 * applications.
	 *
	 * @return uiMessage, returns the instance of {@code UIMessage} class.
	 */
	private UIMessage compareManifestAndFingerPrintInformation() {
		UIMessage uiMessage = null;

		if (this.app1FingerPrint != null && this.app2FingerPrint != null) {

			if (this.app1FingerPrint.equals(this.app2FingerPrint)) {
				uiMessage = new UIMessage(UIMessageType.INFO, "Apps and its versions are the same");
			} else {
				if (this.manifestInfo1 != null && this.manifestInfo2 != null) {
					if (this.manifestInfo1.getVersionCode().equals(this.manifestInfo2.getVersionCode())) {
						uiMessage = new UIMessage(UIMessageType.WARNING, "Apps are different but versions are the same");
					} else {
						uiMessage = new UIMessage(UIMessageType.SUGGESTION, "Apps and its versions are different");
					}
				} else {
					uiMessage = new UIMessage(UIMessageType.WARNING, "Apps are the same but no version information");
				}
			}

		} else {
			uiMessage = new UIMessage(UIMessageType.ERROR, "Missing App information");
		}
		return uiMessage;
	}

}
