/**
 *
 */
package de.upb.pga3.panda2.client.core;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.client.core.datastructures.ExtraAnalysisInput;
import de.upb.pga3.panda2.client.core.datastructures.LevelSpecificMode;
import de.upb.pga3.panda2.client.core.datastructures.UIMessage;
import de.upb.pga3.panda2.client.core.datastructures.UIMessageType;
import de.upb.pga3.panda2.core.Analysis;
import de.upb.pga3.panda2.core.AnalysisFactory;
import de.upb.pga3.panda2.core.AnalysisRunner;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.core.datastructures.Message;
import de.upb.pga3.panda2.extension.lvl1.AnalysisResultLvl1;
import de.upb.pga3.panda2.extension.lvl1.DetailLevelLvl1;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisResultLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisResultLvl2a.DetailLevelLvl2a;
import de.upb.pga3.panda2.extension.lvl2b.DetailLevelLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.ExtraAnalysisInputLvl2b;

/**
 *
 * This abstract class used for providing the common methods and variables which
 * will be used for seperate Business Logic Class for command line and GUI by
 * extending this class.
 *
 * @author Sriram
 */
public abstract class Client {

	static final Logger LOGGER = LogManager.getLogger(Client.class);

	// Variable for the instance of UserInput class which contains the values
	// set by the user.
	private UserInput userInput;
	// Variable for the instance of ConfigManager class.
	// Commented the following variable as we are using static methods of that
	// class.
	// private ConfigManager configManager;
	// Variable for the instance of the AppIdentifier class.
	private AppIdentifier appIdentifier;
	// Variable for maintaining the Analysis class instances as a list.
	private List<Analysis> analysisList;
	// Variable for persisting the AnalysisResult
	private AnalysisResult analysisResult;
	// Variable for the instance of the ResultLoader class.
	private ResultLoader resultLoader;
	// Variable for the instance of the ResultStorer class.
	// Commented the following variable as we are using static methods of that
	// class.
	// private ResultStorer resultStorer;
	/*
	 * Variable for persisting the value of user selected mode as a boolean
	 * value
	 */
	private boolean isComparisonMode = false;

	/**
	 * Constructor without parameter. Used as a Super Constructor without
	 * parameters for the child class constructors.
	 *
	 * It also creates the required user object within the method e.g. the
	 * {@link UserInput}
	 */
	public Client() {
		this.userInput = new UserInput();
	}

	/**
	 * Constructor with parameter. Used as a Super Constructor with parameters
	 * for the child class constructors.
	 *
	 * It also creates the required user object within the method e.g. the
	 * {@link UserInput}
	 *
	 * @param level
	 *            Represents the Level selected by the user.
	 * @param levelSpecificMode
	 *            Represents the mode specific to the level by the user. Value
	 *            of this value will be null for Level1 and Level2a.
	 * @param mode
	 *            Represents the Mode selected by the user.
	 *
	 *
	 */
	/*
	 * public Client(final Level level, final LevelSpecificMode
	 * levelSpecificMode, final Mode mode) { this.userInput = new
	 * UserInput(level, levelSpecificMode, mode); this.isComparisonMode =
	 * mode.equals(Mode.COMPARISON); }
	 */

	/**
	 * Get the instance of the UserInput class.
	 *
	 * @return userInput, which returns the instance of the {@code UserInput}
	 *         class.
	 */
	public UserInput getUserInput() {
		return this.userInput;
	}

	/**
	 * Set the instance of UserInput class.
	 *
	 * @param userInput
	 *            Instance of UserInput
	 * @see UserInput
	 */
	public void setUserInput(final UserInput userInput) {
		this.userInput = userInput;
	}

	/**
	 * Get the instance of the ConfigManager class.
	 *
	 * @return configManager, which returns the instance of the
	 *         {@code ConfigManager} class.
	 *
	 * @see ConfigManager
	 *
	 */
	/*
	 * public ConfigManager getConfigManager() { return this.configManager; }
	 */

	/**
	 * Set the instance of ConfigManager class.
	 *
	 * @param configManager
	 *            Instance of ConfigManager.
	 *
	 */
	/*
	 * public void setConfigManager(final ConfigManager configManager) {
	 * this.configManager = configManager; }
	 */

	/**
	 * Get the instance of the AppIdentifier class.
	 *
	 * @return appIdentifier, which returns the instance of the
	 *         {@code AppIdentifier} class.
	 * @see AppIdentifier
	 */
	/*
	 * public AppIdentifier getAppIdentifier() { return this.appIdentifier; }
	 */

	/**
	 * Set the instance of AppIdentifier class.
	 *
	 * @param appIdentifier
	 *            Instance of AppIdentifier class.
	 *
	 */
	/*
	 * public void setAppIdentifier(final AppIdentifier appIdentifier) {
	 * this.appIdentifier = appIdentifier; }
	 */

	/**
	 * Get the list of instances of Analysis class.
	 *
	 * @return analysisList, A list of {@code Analysis} instances
	 * @see Analysis
	 */
	/*
	 * public List<Analysis> getAnalysisList() { return this.analysisList; }
	 */

	/**
	 * Set the list of instance of the Analysis class.
	 *
	 * @param analysisList
	 *            Instance of the list of Analysis class.
	 */
	/*
	 * public void setAnalysisList(final List<Analysis> analysisList) {
	 * this.analysisList = analysisList; }
	 */

	/**
	 * Get the instance of the AnalysisResult
	 *
	 * @return The instance of the {@code AnalysisResult} class.
	 * @see AnalysisResult
	 */
	public AnalysisResult getAnalysisResult() {
		return this.analysisResult;
	}

	/**
	 * Set the instance of the AnalysisResult class.
	 *
	 * @param analysisResult
	 *            Instance of the AnalysisResult class.
	 */
	/*
	 * public void setAnalysisResult(final AnalysisResult analysisResult) {
	 * this.analysisResult = analysisResult; }
	 */

	/**
	 * Get the instance of the ResultLoader
	 *
	 * @return The instance of the {@code ResultLoader} class.
	 *
	 * @see ResultLoader
	 *
	 */
	public ResultLoader getResultLoader() {
		return this.resultLoader;
	}

	/**
	 * Set the instance of the ResultLoader class.
	 *
	 * @param resultLoader
	 *            Instance of the ResultLoader class.
	 */
	/*
	 * public void setResultLoader(final ResultLoader resultLoader) {
	 * this.resultLoader = resultLoader; }
	 */

	/**
	 * Get the instance of the ResultStorer class.
	 *
	 * @return The instance of the {@code ResultStorer} class.
	 * @see ResultStorer
	 */
	/*
	 * public ResultStorer getResultStorer() { return this.resultStorer; }
	 */

	/**
	 * Set the instance of the ResultStorer class.
	 *
	 * @param resultStorer
	 *            Instance of the ResultStorer class
	 */
	/*
	 * public void setResultStorer(final ResultStorer resultStorer) {
	 * this.resultStorer = resultStorer; }
	 */

	/**
	 * Get the boolean value which represents whether the user selected mode is
	 * comparison or not.
	 *
	 * @return boolean value. True => When user selected the comparison mode
	 */
	public boolean isComparisonMode() {
		return this.isComparisonMode;
	}

	/**
	 * Set the boolean value which represents whether the user selected mode is
	 * comparison or not.
	 *
	 * @param isComparisonMode
	 */
	public void setComparisonMode(final boolean isComparisonMode) {
		this.isComparisonMode = isComparisonMode;
	}

	/**
	 * Method for validating the input and comparing the application.
	 *
	 * @return comparisonUIMessage, represents the instance of the
	 *         {@code UIMessage}
	 */
	public UIMessage compareApp() {
		UIMessage comparisonUIMessage = null;
		try {
			final UIMessage uiMessage = validateInput();
			if (uiMessage != null && !uiMessage.getType().equals(UIMessageType.ERROR)) {
				this.appIdentifier = new AppIdentifier();
				if (this.userInput.getPreviousAnalysisResult() != null) {
					comparisonUIMessage = this.appIdentifier.compareAppWithPreviousResult(
							this.userInput.getInitialInputAPKFiles().get(0),
							this.userInput.getPreviousAnalysisResult());
				} else {
					comparisonUIMessage = this.appIdentifier.compareAppWithPreviousResult(
							this.userInput.getInitialInputAPKFiles().get(0),
							this.userInput.getPreviousAnalysisResultFile());
				}
			} else {
				comparisonUIMessage = new UIMessage(UIMessageType.ERROR, "Unable to validate the User Input");
			}
		} catch (final Exception ex) {
			comparisonUIMessage = new UIMessage(UIMessageType.ERROR, ex.getMessage());
		}
		return comparisonUIMessage;
	}

	/**
	 * Method for performing the analysis.
	 *
	 * @return uiMessage, represents the instance of the {@code UIMessage}
	 */
	public UIMessage performAnalysis() {

		UIMessage uiMessage = null;
		try {
			// For summary mode, validation will be done here.
			if (!isComparisonMode()) {
				uiMessage = validateInput();
			}
			final boolean isValidate = uiMessage != null && !uiMessage.getType().equals(UIMessageType.ERROR);
			if (isValidate || this.isComparisonMode) {
				createAnalysis();
				runAnalysis();
				if (this.analysisResult != null) {
					uiMessage = new UIMessage(UIMessageType.INFO, "Fetched the analysis result successfully");
				} else {
					uiMessage = new UIMessage(UIMessageType.ERROR, "Unable to fetch the analysis result");
				}
			}
		} catch (final Exception ex) {
			ex.printStackTrace();
			LOGGER.debug("Unexpected error during analysis: ", ex);
			uiMessage = new UIMessage(UIMessageType.ERROR, "Unable to fetch the analysis result");

		}
		return uiMessage;
	}

	/**
	 * Method for Saving the Analysis Result.
	 *
	 * @return uiMessage, instance of the UIMessage class.
	 */
	public UIMessage saveResult() {
		return ResultStorer.storeResult(this.analysisResult, this.userInput.getFilePath());

	}

	/**
	 * Method for fetching the filtered textual result as a String. Currently
	 * used as abstract method.
	 *
	 * @param detailLevel
	 *            the instance of implemented classes of the DetailLevel
	 *            interface
	 * @param resultFilter
	 *            Result filters as the instances of the list of String.
	 * @return textualResult, represents the analysis textual result as instance
	 *         of the String class
	 */
	public abstract String filterTextViewResult(final DetailLevel detailLevel, final List<String> resultFilter);

	/**
	 *
	 * @param detailLevel
	 *            the instance of implemented classes of the DetailLevel
	 *            interface
	 * @param resultFilter
	 *            Result filters as the instances of the list of String.
	 * @return graphicalResult, represents the analysis textual result as
	 *         instance of the String class
	 */
	public abstract String filterGraphicalViewResult(final DetailLevel detailLevel, final List<String> resultFilter);

	/**
	 *
	 * This method convert the UIMessage to String value
	 *
	 * @param uiMessage
	 *            the instance of the UIMessage class
	 * @return stringMessage, the message taken from UIMessage object as a
	 *         String instance.
	 */
	public String showUIMessages(final UIMessage uiMessage) {
		String stringMessage = null;
		if (uiMessage != null) {
			stringMessage = uiMessage.getBody();
		}
		return stringMessage;
	}

	/**
	 *
	 * This method convert the Message to String value.
	 *
	 * @param resultMessage
	 *            the instance of the UIMessage class
	 * @return String, the message taken from UIMessage object as a String
	 *         instance.
	 */
	public String showMessages(final Message resultMessage) {
		String stringMessage = null;
		if (resultMessage != null) {
			stringMessage = resultMessage.getBody();
		}
		return stringMessage;
	}

	/**
	 * Method for validating the user input.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link ConfigManager}
	 *
	 * @return uiMessage, which returns the instance of the {@code UIMessage}
	 *         class.
	 */
	private UIMessage validateInput() {
		UIMessage uiMessage = null;
		try {
			this.userInput.createAPKFilesForInitialInput();
			if (this.isComparisonMode) {
				this.userInput.createPreviousAnalysisResultFile();
				final String fileExtension = getFileExtension(this.userInput.getPreviousAnalysisResultFile());
				if (fileExtension.equalsIgnoreCase("pa2")) {
					this.userInput.setPreviousAnalysisResult(loadPreviousAnalysisResult());
				}
			}
			if (AnalysisRegistry.getInstance().getName(this.userInput.getLevel()).equals(AnalysisRegistry.LEVEL2B)) {
				this.userInput.createAPKFilesForNonNativeApps();
			}
			uiMessage = ConfigManager.validateInput(this.userInput);
		} catch (final Exception ex) {
			ex.printStackTrace();
			uiMessage = new UIMessage(UIMessageType.ERROR, ex.getMessage());
		}
		return uiMessage;
	}

	/**
	 *
	 * Method for loading the previously analyzed result for the comparison mode
	 * based on the provided user input.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link ResultLoader}
	 *
	 * @return
	 *
	 */
	private AnalysisResult loadPreviousAnalysisResult() {
		return ResultLoader.loadPreviousAnalysisResult(this.userInput.getSelectedComparisonInput());
	}

	/**
	 *
	 * Method for loading the previously analyzed result for the comparison mode
	 * based on the provided user input.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link ResultLoader}
	 *
	 * @return
	 *
	 */
	public void viewPreviousAnalysisResult() {
		this.analysisResult = ResultLoader.loadPreviousAnalysisResult(this.userInput.getFilePath());
	}

	/**
	 * Method for creating the instances of the list of Analysis class by
	 * calling the necessary methods using the object of other classes.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link AnalysisResultLvl1, AnalysisFactoryLvl2a, AnalysisFactoryLvl1,
	 * AnalysisFactoryLvl2a}
	 *
	 * @return void
	 *
	 */
	private void createAnalysis() {
		final AnalysisResult previousAnalysisResult = this.userInput.getPreviousAnalysisResult();
		final List<File> apkFiles = this.userInput.getInitialInputAPKFiles();
		final File apkFile = apkFiles.get(0);
		final File prevResultFile = this.userInput.getPreviousAnalysisResultFile();

		final String name = AnalysisRegistry.getInstance().getName(this.userInput.getLevel());
		final AnalysisFactory analysisFactory;
		if (this.userInput.getLevelSpecificMode() != null) {
			final ExtraAnalysisInput extraInput = new ExtraAnalysisInputLvl2b(
					this.userInput.getLevelSpecificMode().equals(LevelSpecificMode.ALL));
			analysisFactory = AnalysisRegistry.getInstance().getFactory(name, apkFile, previousAnalysisResult,
					prevResultFile, this.userInput.getNonNativeAPKFiles(), extraInput);
		} else {
			analysisFactory = AnalysisRegistry.getInstance().getFactory(name, apkFile, previousAnalysisResult,
					prevResultFile, this.userInput.getNonNativeAPKFiles(), null);
		}

		this.analysisList = analysisFactory.createAnalysis();
	}

	/**
	 * Method for running the Analysis and for fetching the final analysis
	 * result.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link AnalysisRunner}
	 *
	 * @return void
	 *
	 */
	private void runAnalysis() {
		final AnalysisRunner analysisRunner = new AnalysisRunner();
		this.analysisResult = analysisRunner.analyze(this.analysisList);
	}

	/**
	 * Method for fetching the list of result filters from the analysis result.
	 *
	 * @return instance of List<String>, Result Filters based on the analysis
	 *         result as list of String values.
	 */
	public List<String> getResultFilters() {
		return this.analysisResult.getFilters();
	}

	/**
	 * Method for fetching the list of Detail Levels from the analysis result.
	 *
	 * @return instance of List<String>, Detail Levels based on the analysis
	 *         result as list of String values.
	 *
	 */
	public List<String> getDetailLevels() {
		final List<String> detailLevels = new ArrayList<String>();
		final List<DetailLevel> detailLevelList = this.analysisResult.getDetailLevels();
		for (final DetailLevel detailLevel : detailLevelList) {
			detailLevels.add(detailLevel.toString());
		}
		return detailLevels;
	}

	/**
	 * Method for getting the default DetailLevel instance.
	 *
	 * @return detailLevel, represents the instance of the classes implemented
	 *         using the {@code DetailLevel} interface.
	 *
	 */
	public DetailLevel getDefaultDetailLevel() {
		DetailLevel detailLevel = null;
		if (this.analysisResult instanceof AnalysisResultLvl1) {
			detailLevel = DetailLevelLvl1.APP;
		} else if (this.analysisResult instanceof AnalysisResultLvl2a) {
			detailLevel = DetailLevelLvl2a.RES_TO_RES;
		} else {
			detailLevel = DetailLevelLvl2b.APP;
		}
		return detailLevel;
	}

	/**
	 * Converting the Message to UIMessage instances.
	 *
	 * @param message
	 *            Instance of the {@code Message} class.
	 * @return uiMessage, represents the instance of UIMessage class.
	 */
	/*
	 * private UIMessage convertMessageToUIMessage(final Message message) {
	 * final String uiMessageType = message.getType().toString(); return new
	 * UIMessage(UIMessageType.valueOf(uiMessageType), message.getBody()); }
	 */

	/**
	 * Method for converting the selected DetailLevel from String instance to
	 * DetailLevel instance
	 *
	 * @param selectedDetailLevel
	 *
	 * @return selectedDetailLvl, represents the instance of the classes
	 *         implemented using the {@code DetailLevel} interface.
	 */
	public DetailLevel getDetailLevelFromString(final String selectedDetailLevel) {
		DetailLevel selectedDetailLvl = null;
		if (this.analysisResult instanceof AnalysisResultLvl1) {
			selectedDetailLvl = DetailLevelLvl1.valueOf(selectedDetailLevel.toUpperCase());
		} else if (this.analysisResult instanceof AnalysisResultLvl2a) {
			selectedDetailLvl = DetailLevelLvl2a.valueOf(selectedDetailLevel.toUpperCase());
		} else {
			selectedDetailLvl = DetailLevelLvl2b.valueOf(selectedDetailLevel.toUpperCase());
		}
		return selectedDetailLvl;
	}

	/**
	 * Get the list of messages from the analysis result
	 *
	 * @return the list of the instances of {@code Message} class.
	 */
	public List<Message> getMessages() {
		return this.analysisResult.getMessages();

	}

	/**
	 * Prints the message in the console
	 *
	 * @param msg
	 *            The message to be printed
	 */
	@SuppressWarnings("PMD")
	public static void print(final String msg) {
		System.out.println(msg);
	}

	/**
	 * Method for getting the file extension.
	 *
	 * @param file
	 * @return file extension as String value
	 */
	private static String getFileExtension(final File file) {
		final String fName = file.getName();
		if (fName.lastIndexOf(".") != -1 && fName.lastIndexOf(".") != 0) {
			return fName.substring(fName.lastIndexOf(".") + 1);
		} else {
			return "";
		}
	}
}
