package de.upb.pga3.panda2.core;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.utilities.MemoryUsageMonitor;
import de.upb.pga3.panda2.utilities.StopWatch;

/**
 * Helper class to perform an analysis consisting of a list of {@link Analysis}
 * instances which has been created by an {@link AnalysisFactory}.
 *
 * @author Fabian
 * @author Felix
 */
public class AnalysisRunner {

	static final Logger LOGGER = LogManager.getLogger(AnalysisRunner.class);

	/**
	 * Executes the provided sequence of {@link Analysis} instances. Those may
	 * be of both types initial and aggregation in a senseful order. To do so
	 * the method stores all results to be able to provide them to an
	 * aggregation analysis. The most recently computed result will be returned.
	 *
	 * @param anaList
	 *            A List of {@link Analysis} instances to be executed.
	 * @return The latest {@link AnalysisResult} that was computed.
	 */
	public AnalysisResult analyze(final List<Analysis> anaList) {

		LOGGER.info("Running analysis consiting of {} steps", anaList.size());

		MemoryUsageMonitor.getInstance().setWarnThreshold(5 * 1024);
		MemoryUsageMonitor.getInstance().setSkipSteps(10);
		MemoryUsageMonitor.getInstance().setupMonitoring(0);

		final StopWatch sWatch = new StopWatch();
		sWatch.start();

		List<AnalysisResult> subResList = new ArrayList<>();
		AnalysisResult res;
		AnalysisResult resAgg = null;
		Analysis lastAna = null;
		for (final Analysis ana : anaList) {
			if (!subResList.isEmpty() && ana.getPreviousAnalysisResult() == null) {
				if (lastAna.aggregates()) {
					resAgg = subResList.get(subResList.size() - 1);
					subResList = new ArrayList<>();
				} else if (!ana.aggregates()) {
					ana.setPreviousAnalysisResult(subResList.get(subResList.size() - 1));
				}
			}
			if (ana.aggregates()) {
				ana.provideSubResults(subResList);
				subResList = new ArrayList<>();
				if (resAgg != null) {
					ana.setPreviousAnalysisResult(resAgg);
				}
			}
			lastAna = ana;
			SootAdapter.getInstance().reinitSoot();
			res = ana.doAnalysis();
			subResList.add(res);
		}

		sWatch.stop();

		LOGGER.info("Finished analysis");
		LOGGER.info("Time needed for analysis: {}", sWatch.getTimeTaken().toString());
		LOGGER.info("Max. memory needed for analysis: {}MB", MemoryUsageMonitor.getInstance().getMaxMemSeen());

		return subResList.get(subResList.size() - 1);
	}

	public void analyzeAsync(final List<Analysis> anaList, final AsyncAnalysisResult caller) {
		new Thread() {
			@Override
			public void run() {
				caller.analysisFinished(analyze(anaList));
			}
		}.start();
	}
}
