package de.upb.pga3.panda2.core.services;

import java.util.Collection;
import java.util.Map;

import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import soot.Body;
import soot.SootClass;

/**
 * Implementations of this interface are used to analyze statements in order to
 * find certain characteristics. Mostly this will be done by analyzing a Jimple
 * string.
 *
 * @author Felix
 * @author Fabian
 *
 */
public interface StatementAnalyzer {
	/**
	 * Inspects a given body for defined and started intents, both explicit and
	 * implicit.
	 *
	 * @param body
	 *            The method body to be inspected for intents
	 * @return A collection of intents found containing information for each
	 *         intent if it is started for result and its possible target
	 *         classes
	 */
	public Collection<IntentInformation> getAllIntents(Body inBody, EnhancedInput inEnhancedInput,
			Map<String, GlobalVariable> inMapGlobalVars);

	/**
	 * Inspects a given body for defined and started explicit intents.
	 *
	 * @param body
	 *            The method body to be inspected for explicit intents
	 *
	 * @param inMapGlobalVars
	 *            the map contains all defined global variables
	 *
	 * @return A collection of intents found containing information for each
	 *         intent if it is started for result and its target class
	 */
	public Collection<IntentInformation> getExplicitIntents(Body inBody, EnhancedInput inInput,
			Map<String, GlobalVariable> inMapGlobalVars);

	/**
	 * Process constructors and get all defined global variables
	 *
	 * @param inBody
	 *            the body of a constructor
	 * @param inInput
	 *            the EnhancedInput for getting further information
	 * @param inMapGlobalVars
	 *            the map that contains all global variables
	 */
	public void extractGlobalVars(Body inBody, EnhancedInput inInput, Map<String, GlobalVariable> inMapGlobalVars);

	/**
	 * Inspects a given body for defined and started implicit intents.
	 *
	 * @param body
	 *            The method body to be inspected for intents
	 *
	 * @param inMapGlobalVars
	 *            the map contains all defined global variables
	 *
	 * @return A collection of intents found containing information for each
	 *         intent if it is started for result and its possible target
	 *         classes
	 */
	public Collection<IntentInformation> getImplicitIntents(Body inBody, EnhancedInput inEnhancedInput,
			Map<String, GlobalVariable> inMapGlobalVars);

	/**
	 * get target components or classes by action string
	 *
	 * @param actionStr
	 *            the specified action string
	 * @param ei
	 *            the EnhancedInput that may contain target classes
	 *            corresponding to the action string
	 * @return a collection of target classes
	 */
	public Collection<SootClass> getTargetsByActionString(String actionStr, EnhancedInput ei);
}
