package de.upb.pga3.panda2.core.services;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import pxb.android.axml.AxmlVisitor;
import soot.Scene;
import soot.SceneTransformer;
import soot.SootClass;

/**
 * @author RamKumar
 *
 *         The Implementation of this parser and associated classes is based in
 *         the implementation of FlowDroid
 */

public class XMLLayoutParser extends SceneTransformer {

	private static final Logger LOGGER = LogManager.getLogger(XMLLayoutParser.class);
	private String packagename;

	private String apkName;
	private List<String> entryClasses;
	private final Map<String, List<String>> callbackMethods = new HashMap<String, List<String>>();

	public XMLLayoutParser(final String packageName, final String apkName, final List<String> entryClasses) {
		this.packagename = packageName;
		this.apkName = apkName;
		this.entryClasses = entryClasses;
	}

	@Override
	protected void internalTransform(final String phaseName, final Map<String, String> options) {

		final File apkFile = new File(this.apkName);
		if (!apkFile.exists()) {
			throw new RuntimeException("File " + this.apkName + " doesn't exist.");
		}

		try {
			final ZipFile zFile = new ZipFile(apkFile);
			final Enumeration<?> entries = zFile.entries();
			while (entries.hasMoreElements()) {
				final ZipEntry entry = (ZipEntry) entries.nextElement();
				final String entryName = entry.getName();
				parseLayoutFile(entryName, zFile.getInputStream(entry));
			}

		} catch (final ZipException e) {
			e.printStackTrace();
		} catch (final IOException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Parse the binary layout file
	 *
	 * @param fileName
	 *            layout file name
	 * @param iStream
	 *            input stream
	 */

	public void parseLayoutFile(final String fileName, final InputStream iStream) {

		if (!fileName.startsWith("res/layout")) {
			return;
		}
		if (!fileName.endsWith(".xml")) {
			return;
		}

		String entryClass = fileName.substring(0, fileName.lastIndexOf("."));
		if (!this.packagename.isEmpty()) {
			entryClass = this.packagename + "." + entryClass;
		}

		if (!fileName.startsWith("res/layout")) {
			return;
		}

		try {
			final XMLHandler handler = new XMLHandler(iStream, new LayoutBinaryParser());
			parseLayoutNode(fileName, handler.getDocument().getRootNode());

		} catch (final Exception ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * Parse the root node in layout file
	 *
	 * @param layoutFile
	 *            layout file name
	 * @param rootNode
	 *            layout file's root node
	 */
	private void parseLayoutNode(final String layoutFile, final XMLNode rootNode) {
		if (rootNode.getTag() == null || rootNode.getTag().isEmpty()) {
			return;
		}

		final String tname = rootNode.getTag().trim();
		if (tname.equals("dummy")) {
			// ignore dummy node
		} else if (tname.equals("include")) {
			// ignore this node too
		} else if (tname.equals("merge")) {
			// we skip this node too
		} else if (tname.equals("fragment")) {
			final XMLAttribute<?> attr = rootNode.getAttribute("name");
			if (attr == null) {
				LOGGER.warn("Fragment without class name detected in layout file");
			} else {
				if (attr.getType() != AxmlVisitor.TYPE_STRING) {
					LOGGER.warn(
							"Invalid targer resource " + attr.getValue() + "for fragment class value in lazout file");
				}
				getLayoutClass(attr.getValue().toString());
			}
		} else {
			parseLayoutAttributes(layoutFile, null, rootNode);

		}

		// Parse the child nodes
		for (final XMLNode childNode : rootNode.getChildren()) {
			parseLayoutNode(layoutFile, childNode);
		}
	}

	/**
	 * Parser layout attributes associated with layout node
	 *
	 * @param layoutFile
	 *            layout file name
	 * @param childClass
	 *            child classes of root node
	 * @param rootNode
	 *            root node
	 */

	private void parseLayoutAttributes(final String layoutFile, final SootClass childClass, final XMLNode rootNode) {

		for (final Entry<String, XMLAttribute<?>> entry : rootNode.getAttributes().entrySet()) {
			final String attName = entry.getKey().trim();
			final XMLAttribute<?> attr = entry.getValue();

			if (attName.isEmpty()) {
				continue;
			} else if (attName.equals("onClick") && attr.getType() == AxmlVisitor.TYPE_STRING
					&& attr.getValue() instanceof String) {
				final String strData = ((String) attr.getValue()).trim();
				addCallbackMethod(layoutFile, strData);
			}

		}

	}

	/**
	 * Add identified call back methods to map
	 *
	 * @param layoutFileName
	 *            layout file name
	 * @param callbackMethod
	 *            call back method name
	 */
	private void addCallbackMethod(final String layoutFileName, final String callbackMethod) {

		if (this.callbackMethods.containsKey(layoutFileName)) {
			this.callbackMethods.get(layoutFileName).add(callbackMethod);
		} else {
			final List<String> callbacks = new ArrayList<>();
			callbacks.add(callbackMethod);
			this.callbackMethods.put(layoutFileName, callbacks);
		}
	}

	/**
	 * Get Layout's class from Soot
	 *
	 * @param className
	 *            class name
	 * @return SootClass for the class name
	 */

	private SootClass getLayoutClass(String className) {
		// Cut off some junk returned by the parser
		if (className.startsWith(";")) {
			className = className.substring(1);
		}

		if (className.contains("(") || className.contains("<") || className.contains("/")) {

			return null;
		}

		SootClass sc = Scene.v().forceResolve(className, SootClass.BODIES);
		if ((sc == null || sc.isPhantom()) && !this.packagename.isEmpty()) {
			sc = Scene.v().forceResolve(this.packagename + "." + className, SootClass.BODIES);
		}
		if (sc == null || sc.isPhantom()) {
			sc = Scene.v().forceResolve("android.view." + className, SootClass.BODIES);
		}
		if (sc == null || sc.isPhantom()) {
			sc = Scene.v().forceResolve("android.widget." + className, SootClass.BODIES);
		}
		if (sc == null || sc.isPhantom()) {
			sc = Scene.v().forceResolve("android.webkit." + className, SootClass.BODIES);
		}
		if (sc == null || sc.isPhantom()) {
			LOGGER.info("Could not find layout class " + className);
			return null;
		}
		return sc;
	}

	/**
	 * Returns the call back map
	 *
	 * @return call back methods map
	 */

	public Map<String, List<String>> getLayoutCallbackMethods() {
		return this.callbackMethods;
	}

}
