package de.upb.pga3.panda2.extension;

import java.nio.file.Path;
import java.util.List;

import de.upb.pga3.panda2.core.AnalysisProcedure;
import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.Input;

/**
 * Performs initial and aggregation analyses based on the three steps
 * {@link Enhancer}, {@link GraphGenerator} and {@link Analyzer}.
 *
 * @author Fabian
 *
 */
public class A3AnalysisProcedure implements AnalysisProcedure {

	private final GraphGenerator graphGenerator;
	private final Analyzer analyzer;

	/**
	 * Creates an instance of {@code A3AnalysisProcedure} with the given
	 * {@link GraphGenerator} and {@link Analyzer}.
	 *
	 * @param gen
	 *            The {@code GraphGenerator} to be used for analysis.
	 * @param an
	 *            The {@code Analyzer} to be used for analysis.
	 */
	public A3AnalysisProcedure(final GraphGenerator gen, final Analyzer an) {
		this.graphGenerator = gen;
		this.analyzer = an;
	}

	@Override
	public AnalysisResult doAggregationAnalysis(final List<AnalysisResult> subResList, final AnalysisResult prevRes) {

		final Input in = new Enhancer().collectResults(subResList);
		return continueAnalysis(in, prevRes, null);
	}

	@Override
	public AnalysisResult doInitialAnalysis(final Path apkFile, final AnalysisResult prevRes) {

		final Input in = new Enhancer().enhance(apkFile);
		return continueAnalysis(in, prevRes, apkFile);
	}

	/**
	 * Continues the analysis after the {@link Input} was created.
	 *
	 * @param in
	 *            The {@code Input}.
	 * @param prevRes
	 *            The result of a previous analysis for {@code COMPARISON} mode.
	 * @param apk
	 *            The apk file for initial type of analysis
	 * @return The result computed by {@link GraphGenerator} and
	 *         {@link Analyzer}.
	 */
	private AnalysisResult continueAnalysis(final Input in, final AnalysisResult prevRes, final Path apk) {
		final AnalysisGraph aGraph = this.graphGenerator.generateGraph(in, apk);
		return this.analyzer.analyze(aGraph, prevRes);
	}

}
