package de.upb.pga3.panda2.extension;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.SootAdapter;
import de.upb.pga3.panda2.core.SootAdapter.SootPhase;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Input;
import de.upb.pga3.panda2.core.datastructures.ResultInput;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.core.services.GlobalVariable;
import de.upb.pga3.panda2.core.services.StatementAnalyzer;
import de.upb.pga3.panda2.core.services.XMLParser;
import de.upb.pga3.panda2.utilities.Constants;
import soot.Body;
import soot.PatchingChain;
import soot.Scene;
import soot.SootClass;
import soot.SootMethod;
import soot.Unit;

/**
 * Class for creating an {@link Input} from an apk file or a sequence of
 * sub-results. This is the first step of the {@link A3AnalysisProcedure}.
 *
 * @author Fabian
 * @author Felix
 * @author nptsy
 */
public class Enhancer {
	// Logger initialization
	private static final Logger LOGGER = LogManager.getLogger(Enhancer.class);
	boolean librariesWarning = true;

	/**
	 * Creates an {@link EnhancedInput} out of the given apk file.
	 *
	 * @param apk
	 *            The apk file to create the {@code EnhancedInput} for.
	 * @return The {@code EnhancedInput} for the given apk file.
	 */
	public Input enhance(final Path apk) {
		LOGGER.info("Enhancer started (" + apk.toString() + ")");

		SootAdapter.getInstance().init(apk);
		SootAdapter.getInstance().run(SootPhase.WJTP);

		final XMLParser xmlParser = CoreServices.getXMLParserInstance();
		xmlParser.fetchData(apk.toAbsolutePath().toString());
		final String appName = xmlParser.getAppName();

		final EnhancedInput ei = new EnhancedInput(appName, xmlParser.getManifestInformation());
		ei.setScene(Scene.v());

		final PermissionMapper permMapper = new PermissionMapper(ei);

		boolean isAndroidComponent = false;
		final StatementAnalyzer stmtAnalyzer = CoreServices.getStatementAnalyzerInstance();

		// map that contains global variables of a class
		final Map<String, GlobalVariable> mapGlobalVars = new HashMap<>();

		final SootClass[] classes = ei.getAppClasses();

		for (final SootClass c : classes) {
			checkForComponent(ei, c, permMapper);
		}

		for (final SootClass c : classes) {
			isAndroidComponent = ei.isAndroidComponent(c);
			final List<SootMethod> methods = c.getMethods();

			// in case of android component
			if (isAndroidComponent) {
				for (final SootMethod m : methods) {
					try {
						if (isConstructor(m)) {
							final Body body = m.retrieveActiveBody();
							stmtAnalyzer.extractGlobalVars(body, ei, mapGlobalVars);
						}
					} catch (final RuntimeException e) {
						LOGGER.warn(e.getMessage());
					}
				}
			}

			for (final SootMethod m : methods) {
				try {
					final Body body = m.retrieveActiveBody();

					final PatchingChain<Unit> units = body.getUnits();
					for (final Unit u : units) {
						ei.mapUnitToBody(u, body);
						ei.mapUnitByStrToPkgStr(c.toString(), c.getJavaPackageName());
						permMapper.mapPermissionsToStatement(u);
					}

					// Fill intent lists
					if (isAndroidComponent) {
						ei.addIntents(stmtAnalyzer.getAllIntents(body, ei, mapGlobalVars));
					}
				} catch (final RuntimeException e) {
					if (e.getMessage().contains("No method source set for method")) {
						if (this.librariesWarning) {
							this.librariesWarning = false;
							LOGGER.warn("App contains external libraries. Library classes will be skipped.");
						}
					} else {
						LOGGER.warn(e.getMessage());
					}
				}
			}
		}

		permMapper.mapPermissionsToApp();

		LOGGER.info("Enhancer finished (" + apk.toString() + ").");
		return ei;
	}

	private static boolean checkForComponent(final EnhancedInput ei, final SootClass c,
			final PermissionMapper permMapper) {
		try {
			SootClass superClass = c.getSuperclass();
			while (superClass != null) {
				final String superClassName = superClass.getName();
				if (superClassName.equals(Constants.ANDROID_APP_ACTIVITY)
						|| superClassName.equals(Constants.ANDROID_APP_SERVICE)
						|| superClassName.equals(Constants.ANDROID_APP_RECEIVER)
						|| superClassName.equals(Constants.ANDROID_APP_PROVIDER)) {
					final Set<String> providerURLs;
					if (superClassName.equals(Constants.ANDROID_APP_PROVIDER)) {
						providerURLs = new HashSet<>(0);
						providerURLs.add(CoreServices.getXMLParserInstance().getContentProviderURIs().get(c.getName()));
					} else {
						providerURLs = null;
					}

					ei.setAsAndroidComponent(c, providerURLs);
					permMapper.mapPermissionsToComponent(c);
					return true;
				}
				superClass = superClass.getSuperclass();
			}
		} catch (final RuntimeException e) {
			// No superclass => No android component
			return false;
		}

		return false;
	}

	/**
	 * Creates a {@link ResultInput} from a list of given {@link AnalysisResult}
	 * s
	 *
	 * @param results
	 *            The list of {@code AnalysisResult}s.
	 * @return The {@code ResultInput} containing the given results.
	 */
	public Input collectResults(final List<AnalysisResult> results) {
		LOGGER.info("Enhancer is collecting results for aggregation analysis.");
		return new ResultInput(results);
	}

	/**
	 * is method a constructor
	 *
	 * @param inSootMethod
	 * @return true if the method is a constructor otherwise false
	 */
	private static boolean isConstructor(final SootMethod inSootMethod) {
		if (inSootMethod != null) {
			final String signature = inSootMethod.getSignature();
			if (signature.contains(Constants.KEY_WORD_CONSTRUCTOR)) {
				return true;
			}
		}
		return false;
	}
}
