package de.upb.pga3.panda2.extension.lvl1;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import de.upb.pga3.panda2.core.Analysis;
import de.upb.pga3.panda2.core.AnalysisFactory;
import de.upb.pga3.panda2.core.AnalysisProcedure;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.extension.A3AnalysisProcedure;
import de.upb.pga3.panda2.extension.lvl1.analyzer.AnalyzerLvl1;
import de.upb.pga3.panda2.extension.lvl1.graphgenerator.GraphGeneratorLvl1;

/**
 * This is the {@link AnalysisFactory} for generating any type of level 1
 * analysis.
 *
 * @author Felix
 */
public class AnalysisFactoryLvl1 implements AnalysisFactory {
	private final File apk;
	private final File apkForPrevRes;
	private final AnalysisResult prevRes;

	/**
	 * Constructor for the level 1 factory. Providing the parameters will be
	 * enough to create any type of level 1 analysis.
	 *
	 * @param apk
	 *            Input .apk file belonging to the App that is going to be
	 *            analyzed.
	 * @param prevRes
	 *            In comparison mode this parameter will contain the analysis
	 *            result of a previously executed level 1 analysis.
	 */
	public AnalysisFactoryLvl1(final File apk, final AnalysisResultLvl1 prevRes) {
		this.apk = apk;
		this.prevRes = prevRes;
		this.apkForPrevRes = null;
	}

	/**
	 * Constructor for the level 1 factory. Providing the parameters will be
	 * enough to create any type of level 1 analysis.
	 *
	 * @param apk
	 *            Input .apk file belonging to the App that is going to be
	 *            analyzed.
	 * @param prevRes
	 *            .apk file that will be used to compute the previous result
	 *            which is then in turn compared to the analyzed App provided by
	 *            the apk parameter.
	 */
	public AnalysisFactoryLvl1(final File apk, final File apkForPrevRes) {
		this.apk = apk;
		this.prevRes = null;
		this.apkForPrevRes = apkForPrevRes;
	}

	/**
	 * This method generates a list of analyses to be executed in order to run a
	 * level 1 analysis. It creates all required objects within e.g. the
	 * {@link AnalysisProcedure}.
	 *
	 * @return The created list of analyses to be executed.
	 */
	@Override
	public List<Analysis> createAnalysis() {
		// Create Analysis
		final AnalysisProcedure anaProc = new A3AnalysisProcedure(new GraphGeneratorLvl1(), new AnalyzerLvl1());
		final Analysis ana = new Analysis(this.apk.toPath(), this.prevRes, anaProc);

		// Create and fill list
		final List<Analysis> analysesList = new ArrayList<>();
		if (this.apkForPrevRes != null) {
			// Create comparison Analysis
			final AnalysisProcedure anaProcComp = new A3AnalysisProcedure(new GraphGeneratorLvl1(), new AnalyzerLvl1());
			final Analysis anaComp = new Analysis(this.apkForPrevRes.toPath(), null, anaProcComp);
			analysesList.add(anaComp);
		}
		analysesList.add(ana);

		return analysesList;
	}
}
