package de.upb.pga3.panda2.extension.lvl1;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.utilities.Constants;
import de.upb.pga3.panda2.utilities.HTMLFrameBuilder;

/**
 * This is the datastructure used for any Level 1 analysis result. To do so, it
 * is extending the class {@link AnalysisResult}.
 *
 * @author Felix
 *
 */
public class AnalysisResultLvl1 extends AnalysisResult {
	private static final Logger LOGGER = LogManager.getLogger(AnalysisResultLvl1.class);

	// the analysis graph
	private AnalysisGraph mAnalysisGraph;

	// AnalysisResultLvl1 datastructures
	private ResultLeafLvl1 app;
	private ResultTreeLvl1 components;
	private ResultTreeLvl1 classes;
	private ResultTreeLvl1 methods;

	// Previous result
	private AnalysisResultLvl1 prevResult;
	private Map<ResultLeafLvl1, ResultLeafLvl1> mapCurrentToPrev;

	// List of filters
	private final List<String> filters;
	private List<String> activeFilters;

	// permission uses
	private int[] permissionUses = null;
	private int[] permissionUsesAll = null;
	private int newPermissions, changedPermissions, unchangedPermissions;

	// Graphic settings
	final int height = 1000;
	public static int barWidth = 75;
	final int permissionsStart = 250;
	final int permissionsWidth = 275;
	final int barsFirst = 0;
	final int width = 2 * this.permissionsStart + this.permissionsWidth;
	final int barsSecond = this.width - AnalysisResultLvl1.barWidth;

	// Map of permissions
	Map<Permission, SVGPermissionLvl1> mapOfPermissions = new HashMap<>();

	// Files
	private File htmlFileTextual;
	private File htmlFileGraphical;

	// Tree option
	public static final String TREEOPTION = "Expandable tree";

	public AnalysisResultLvl1(final AnalysisGraph inGraph) {
		setResultIsFile(true);

		this.mAnalysisGraph = inGraph;

		// Files
		this.htmlFileTextual = new File("data/temp/Level1_Result_textual.html");
		this.htmlFileGraphical = new File("data/temp/Level1_Result_graphical.html");

		// Initially mode is summary
		this.prevResult = null;

		// Initializing filterlist
		this.filters = new ArrayList<>();
		this.filters.add(AnalysisResultLvl1.TREEOPTION);
		this.filters.add(ResultTypeLvl1.getStringOfType(ResultTypeLvl1.REQUIRED));
		this.filters.add(ResultTypeLvl1.getStringOfType(ResultTypeLvl1.MAYBE_REQUIRED));
		this.filters.add(ResultTypeLvl1.getStringOfType(ResultTypeLvl1.UNUSED));
		this.filters.add(ResultTypeLvl1.getStringOfType(ResultTypeLvl1.MISSING));
		this.filters.add(ResultTypeLvl1.getStringOfType(ResultTypeLvl1.MAYBE_MISSING));
	}

	/**
	 * The next 5 methods are simple setter methods for parts of the overall
	 * result. The parts can be differentiated by the detail level and the group
	 * they belong to.
	 */
	public void setApp(final ResultLeafLvl1 result) {
		this.app = result;
		updateFilters(result);
	}

	public ResultLeafLvl1 getApp() {
		return this.app;
	}

	public void setComponents(final ResultTreeLvl1 result) {
		this.components = result;
		updateFilters(result);
	}

	public ResultTreeLvl1 getComponents() {
		return this.components;
	}

	public void setClasses(final ResultTreeLvl1 result) {
		this.classes = result;
		updateFilters(result);
	}

	public ResultTreeLvl1 getClasses() {
		return this.classes;
	}

	public void setMethods(final ResultTreeLvl1 result) {
		this.methods = result;
		updateFilters(result);
	}

	public ResultTreeLvl1 getMethods() {
		return this.methods;
	}

	public AnalysisGraph getAnalysisGraph() {
		return this.mAnalysisGraph;
	}

	private int[] getPermissionUses(final DetailLevel inDetailLvl, final List<String> inFilters) {
		getTextualResult(inDetailLvl, inFilters, true);
		return this.permissionUses;
	}

	private void updateFilters(final ResultTreeLvl1 result) {
		if (result != null) {
			for (final ResultLeafLvl1 leaf : result.getLeafs().values()) {
				updateFilters(leaf);
			}
		}
	}

	private void updateFilters(final ResultLeafLvl1 result) {
		if (result != null && !result.getPermissions(ResultTypeLvl1.ALL).isEmpty()) {
			for (final Permission permission : result.getPermissions(ResultTypeLvl1.ALL)) {
				boolean add = true;
				for (final String filter : this.filters) {
					if (filter.equals(Constants.PREFIX_PERMISSION + " " + permission.getName())) {
						add = false;
						break;
					}
				}
				if (add) {
					this.filters.add(Constants.PREFIX_PERMISSION + " " + permission.getName());
				}
			}
		}
	}

	/**
	 * This method will be called once a comparison result is created. It will
	 * bind elements from the analyzed App to elements from the previously
	 * analyzed App.
	 *
	 * @param prevRes
	 *            This parameter contains the previously computed result.
	 */
	public void compare(final AnalysisResult prevRes) {
		// Setting mode to comparison
		this.prevResult = (AnalysisResultLvl1) prevRes;
		this.mapOfPermissions = this.prevResult.getMapOfPermissions();
		this.mapCurrentToPrev = new HashMap<>();

		// App
		this.mapCurrentToPrev.put(this.app, this.prevResult.getApp());

		// Classes
		for (final ResultLeafLvl1 currentLeaf : this.classes.getLeafsAsList()) {
			for (final ResultLeafLvl1 prevLeaf : this.prevResult.getClasses().getLeafsAsList()) {
				if (currentLeaf.getName().equals(prevLeaf.getName())) {
					this.mapCurrentToPrev.put(currentLeaf, prevLeaf);
				}
			}
		}

		// Components
		for (final ResultLeafLvl1 currentLeaf : this.components.getLeafsAsList()) {
			for (final ResultLeafLvl1 prevLeaf : this.prevResult.getComponents().getLeafsAsList()) {
				if (currentLeaf.getName().equals(prevLeaf.getName())) {
					this.mapCurrentToPrev.put(currentLeaf, prevLeaf);
				}
			}
		}

		// Methods
		for (final ResultLeafLvl1 currentLeaf : this.methods.getLeafsAsList()) {
			for (final ResultLeafLvl1 prevLeaf : this.prevResult.getMethods().getLeafsAsList()) {
				if (currentLeaf.getName().equals(prevLeaf.getName())) {
					this.mapCurrentToPrev.put(currentLeaf, prevLeaf);
				}
			}
		}
	}

	@Override
	public String getGraphicalResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {
		try {
			// Reset output file
			resetFile(this.htmlFileGraphical);

			// Setup
			final BufferedWriter bw = new BufferedWriter(new FileWriter(this.htmlFileGraphical));
			final HTMLFrameBuilder fb = new HTMLFrameBuilder(bw,
					"Graphical result - Intra-App Permission Usage Analysis");
			fb.setHeaderAutoHide(inShowStats);
			fb.setHintPersistent(false);
			fb.setCustomStyle(
					"div.icon { float: left; border: 1px solid #000000; width: 12px; height: 7px;	margin-right: 5px; margin-top: 3px;	}");

			// Legend
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#007f22;\"></div>", "REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#0006fd;\"></div>", "MAYBE REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#927a3a;\"></div>", "UNUSED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff7500;\"></div>", "MAYBE MISSING");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff0000;\"></div>", "MISSING");

			// Content
			fb.append("<center>");
			if (inDetailLvl == DetailLevelLvl1.APP) {
				if (this.prevResult != null) {
					getGraphic(this.app, this.prevResult.getApp(), fb);
				} else {
					getGraphic(this.app, null, fb);
				}
			} else if (inDetailLvl == DetailLevelLvl1.COMPONENT) {
				for (final ResultLeafLvl1 leaf : this.components.getLeafsAsList()) {
					if (leaf != null) {
						if (this.prevResult != null) {
							final ResultLeafLvl1 compareLeaf = this.prevResult.getComponents().getLeaf(leaf.getName());
							if (compareLeaf != null) {
								getGraphic(leaf, compareLeaf, fb);
							} else {
								getGraphic(leaf, null, fb);
							}
						} else {
							getGraphic(leaf, null, fb);
						}
					}
				}
			} else {
				fb.append("Graphical result only available in APP and COMPONENT mode. Please choose one of those.");
			}
			fb.append("</center>");

			// Statistics
			final String statsLine = "<strong>"
					+ (this.permissionUsesAll[ResultTypeLvl1.REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl1.UNUSED]
							+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_MISSING]
							+ this.permissionUsesAll[ResultTypeLvl1.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl1.getColorOfType(ResultTypeLvl1.REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.UNUSED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.UNUSED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.MISSING] + "</font>)";
			fb.addStaticticsRow("Analysed permission uses:", statsLine);
			fb.setAppTrustworthy(this.permissionUsesAll[ResultTypeLvl1.MISSING]
					+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_MISSING] == 0);
			fb.complete();
			bw.close();
		} catch (final IOException e) {
			LOGGER.error("Could not access outputfile.");
		}

		return this.htmlFileGraphical.toString();
	}

	private void getGraphic(final ResultLeafLvl1 leaf, final ResultLeafLvl1 compLeaf, final HTMLFrameBuilder fb)
			throws IOException {
		// Reset permission map
		this.mapOfPermissions.clear();

		// Generate SVG items
		final SVGGraphicLvl1 svgGraphic = getGraphic(leaf, compLeaf, this.permissionsStart, this.barsFirst,
				this.permissionsStart + this.permissionsWidth, this.barsSecond);

		// Build
		if (this.prevResult != null) {
			fb.append("<table width=\"" + this.width + "\"><tr><td>Current Result</td><td align=\"center\"><strong>"
					+ leaf.getName() + "</strong></td><td align=\"right\">Previous Result</td></tr></table>\n");
		} else {
			fb.append("<strong>" + leaf.getName() + "</strong><br />\n");
		}
		fb.append("<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"" + this.width + "\" height=\""
				+ (13 + 20 * this.mapOfPermissions.values().size() + 2)
				+ "\" xmlns:xlink=\"http://www.w3.org/1999/xlink\">");
		fb.append(svgGraphic.toString());
		fb.append("</svg><br /><br />");
	}

	private SVGGraphicLvl1 getGraphic(final ResultLeafLvl1 leaf, final ResultLeafLvl1 compLeaf, final int xLineStart,
			final int xLineEnd, final int xLineStartComp, final int xLineEndComp) {
		final int[] permissionUsesGraphical = new int[5];
		final int[] permissionUsesGraphicalComp = new int[5];
		final List<SVGArrowLvl1> arrows = new ArrayList<>();
		final SVGBarLvl1[] bars = new SVGBarLvl1[5];
		int yBars = 0;
		final SVGBarLvl1[] barsComp = new SVGBarLvl1[5];
		int yBarsComp = 0;
		int yPermissions = 13;

		for (int i = 0; i < 5; i++) {
			// Current result
			if (leaf.getPermissions(i) != null) {
				for (final Permission permission : leaf.getPermissions(i)) {
					if (this.activeFilters.contains(ResultTypeLvl1.getStringOfType(i))
							|| this.activeFilters.contains(Constants.PREFIX_PERMISSION + " " + permission.getName())) {
						permissionUsesGraphical[i]++;

						if (!this.mapOfPermissions.containsKey(permission)) {
							this.mapOfPermissions.put(permission, new SVGPermissionLvl1(permission, yPermissions));
							yPermissions += 20;
						}
					}
				}
			}

			// Previous result
			if (compLeaf != null) {
				if (compLeaf.getPermissions(i) != null) {
					for (final Permission permission : compLeaf.getPermissions(i)) {
						if (this.activeFilters.contains(ResultTypeLvl1.getStringOfType(i)) || this.activeFilters
								.contains(Constants.PREFIX_PERMISSION + " " + permission.getName())) {
							permissionUsesGraphicalComp[i]++;

							if (!this.mapOfPermissions.containsKey(permission)) {
								this.mapOfPermissions.put(permission, new SVGPermissionLvl1(permission, yPermissions));
								yPermissions += 20;
							}
						}
					}
				}
			}
		}

		final int sumPermissions = permissionUsesGraphical[ResultTypeLvl1.REQUIRED]
				+ permissionUsesGraphical[ResultTypeLvl1.MAYBE_REQUIRED]
				+ permissionUsesGraphical[ResultTypeLvl1.UNUSED] + permissionUsesGraphical[ResultTypeLvl1.MAYBE_MISSING]
				+ permissionUsesGraphical[ResultTypeLvl1.MISSING];

		for (int i = 0; i < 5; i++) {
			final int barHeight = Math
					.round((float) permissionUsesGraphical[i] / sumPermissions * (13 + 20 * sumPermissions));
			bars[i] = new SVGBarLvl1(xLineEnd, yBars, barHeight);
			yBars = yBars + barHeight;
			if (compLeaf != null) {
				final int barHeightComp = Math
						.round((float) permissionUsesGraphicalComp[i] / sumPermissions * (13 + 20 * sumPermissions));
				barsComp[i] = new SVGBarLvl1(xLineEndComp, yBarsComp, barHeightComp);
				yBarsComp = yBarsComp + barHeightComp;
			}

			// Current result
			if (leaf.getPermissions(i) != null) {
				for (final Permission permission : leaf.getPermissions(i)) {
					if (this.activeFilters.contains(ResultTypeLvl1.getStringOfType(i))
							|| this.activeFilters.contains(Constants.PREFIX_PERMISSION + " " + permission.getName())) {
						final int yLine = this.mapOfPermissions.get(permission).getY() - 4;
						final int[] offset = new int[] { -5, AnalysisResultLvl1.barWidth };
						arrows.add(new SVGArrowLvl1(new int[] { xLineStart + offset[0], yLine },
								new int[] { xLineEnd + offset[1], bars[i].getClosestTo(yLine) }));
					}
				}
			}

			// Previous result
			if (compLeaf != null) {
				if (compLeaf.getPermissions(i) != null) {
					for (final Permission permission : compLeaf.getPermissions(i)) {
						if (this.activeFilters.contains(ResultTypeLvl1.getStringOfType(i)) || this.activeFilters
								.contains(Constants.PREFIX_PERMISSION + " " + permission.getName())) {
							final int yLine = this.mapOfPermissions.get(permission).getY() - 4;
							final int[] offset = new int[] { 5, 0 };
							arrows.add(new SVGArrowLvl1(new int[] { xLineStartComp + offset[0], yLine },
									new int[] { xLineEndComp + offset[1], barsComp[i].getClosestTo(yLine) }));
						}
					}
				}
			}
		}

		return new SVGGraphicLvl1(arrows, this.mapOfPermissions, bars, barsComp, xLineStart, compLeaf != null);
	}

	@Override
	public String getTextualResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {
		// Counters
		this.permissionUses = new int[5];
		this.newPermissions = 0;
		this.changedPermissions = 0;
		this.unchangedPermissions = 0;

		int[] prevPermissionUses = null;
		if (this.prevResult != null) {
			prevPermissionUses = this.prevResult.getPermissionUses(inDetailLvl, inFilters);
		}

		// Assign filters
		if (inFilters == null) {
			this.activeFilters = getFilters();
			this.activeFilters.remove(AnalysisResultLvl1.TREEOPTION);
		} else {
			this.activeFilters = inFilters;
		}

		try {
			// Reset output file
			resetFile(this.htmlFileGraphical);

			// Setup
			final BufferedWriter bw = new BufferedWriter(new FileWriter(this.htmlFileTextual));
			final HTMLFrameBuilder fb = new HTMLFrameBuilder(bw,
					"Textual result - Intra-App Permission Usage Analysis");
			fb.setHeaderAutoHide(inShowStats);
			fb.setHintPersistent(false);
			fb.setCustomStyle(
					"div.icon { float: left; border: 1px solid #000000; width: 12px; height: 7px;	margin-right: 5px; margin-top: 3px;	} div.plus { background: #C9C9C9; float: left; border: 1px outset #CCCCCC; padding: 0px 0px 0px 0px; margin-top: 2px; margin-right: 10px; height: 9px; width: 9px; } div.inner { margin-top:-2px; font-size: 10px; text-align: center; cursor: pointer;	} div.spacer { float: left;	margin-top: 3px; margin-right: 10px; height: 11px; width: 11px;	} div.popout { padding-left: 25px; visibility: hidden; max-height: 0px;	overflow: hidden; }	div.premissions { padding-left: 21px; }	table {	border-spacing:0px;	} td { padding: 0px; }");
			fb.setCustomScript(
					"function show(caller) {	if(caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.visibility == 'visible') { caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.maxHeight='0px'; caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.visibility='hidden';	caller.firstElementChild.innerHTML='+';	caller.style.border='1px outset #CCCCCC'; } else { caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.maxHeight='99999px';	caller.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.nextElementSibling.style.visibility='visible'; caller.firstElementChild.innerHTML='-'; caller.style.border='1px inset #CCCCCC'; }}");

			// Legend
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#007f22;\"></div>", "REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#0006fd;\"></div>", "MAYBE REQUIRED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#927a3a;\"></div>", "UNUSED");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff7500;\"></div>", "MAYBE MISSING");
			fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff0000;\"></div>", "MISSING");

			// Content
			if (inDetailLvl == DetailLevelLvl1.CLASS) {
				getTextualForTree(this.classes, inDetailLvl, fb);
			} else if (inDetailLvl == DetailLevelLvl1.COMPONENT) {
				getTextualForTree(this.components, inDetailLvl, fb);
			} else if (inDetailLvl == DetailLevelLvl1.METHOD) {
				getTextualForTree(this.methods, inDetailLvl, fb);
			} else {
				getTextualForLeaf(this.app, getChildren(this.app, inDetailLvl), inDetailLvl, fb);
			}

			// Statistics
			if (inDetailLvl == DetailLevelLvl1.APP && this.permissionUsesAll == null) {
				this.permissionUsesAll = this.permissionUses;
			} else if (inDetailLvl != DetailLevelLvl1.APP && this.permissionUsesAll == null) {
				getTextualResult(DetailLevelLvl1.APP, inFilters, inShowStats);
			}

			String statsLine = "<strong>"
					+ (this.permissionUsesAll[ResultTypeLvl1.REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_REQUIRED]
							+ this.permissionUsesAll[ResultTypeLvl1.UNUSED]
							+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_MISSING]
							+ this.permissionUsesAll[ResultTypeLvl1.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl1.getColorOfType(ResultTypeLvl1.REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_REQUIRED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.UNUSED) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.UNUSED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MISSING) + "\">"
					+ this.permissionUsesAll[ResultTypeLvl1.MISSING] + "</font>)";
			fb.addStaticticsRow("Analysed permission uses:", statsLine);
			fb.addStaticticsRow("<br />", "<br />");
			fb.addStaticticsRow("<strong>Filtered</strong>", "");
			if (this.prevResult != null) {
				statsLine = "<strong>" + this.newPermissions + "</strong> / <strong>" + this.unchangedPermissions
						+ "</strong> / <strong>" + this.changedPermissions + "</strong>";
				fb.addStaticticsRow("New/Un-/Changed permissions:", statsLine);

				statsLine = "<strong>" + (prevPermissionUses[ResultTypeLvl1.REQUIRED]
						+ prevPermissionUses[ResultTypeLvl1.MAYBE_REQUIRED] + prevPermissionUses[ResultTypeLvl1.UNUSED]
						+ prevPermissionUses[ResultTypeLvl1.MAYBE_MISSING] + prevPermissionUses[ResultTypeLvl1.MISSING])
						+ "</strong> (<font color=\"" + ResultTypeLvl1.getColorOfType(ResultTypeLvl1.REQUIRED) + "\">"
						+ prevPermissionUses[ResultTypeLvl1.REQUIRED] + "</font> / <font color=\""
						+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_REQUIRED) + "\">"
						+ prevPermissionUses[ResultTypeLvl1.MAYBE_REQUIRED] + "</font> / <font color=\""
						+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.UNUSED) + "\">"
						+ prevPermissionUses[ResultTypeLvl1.UNUSED] + "</font> / <font color=\""
						+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_MISSING) + "\">"
						+ prevPermissionUses[ResultTypeLvl1.MAYBE_MISSING] + "</font> / <font color=\""
						+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MISSING) + "\">"
						+ prevPermissionUses[ResultTypeLvl1.MISSING] + "</font>) &#8594; ";
			} else {
				statsLine = "";
			}
			statsLine += "<strong>"
					+ (this.permissionUses[ResultTypeLvl1.REQUIRED] + this.permissionUses[ResultTypeLvl1.MAYBE_REQUIRED]
							+ this.permissionUses[ResultTypeLvl1.UNUSED]
							+ this.permissionUses[ResultTypeLvl1.MAYBE_MISSING]
							+ this.permissionUses[ResultTypeLvl1.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl1.getColorOfType(ResultTypeLvl1.REQUIRED) + "\">"
					+ this.permissionUses[ResultTypeLvl1.REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_REQUIRED) + "\">"
					+ this.permissionUses[ResultTypeLvl1.MAYBE_REQUIRED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.UNUSED) + "\">"
					+ this.permissionUses[ResultTypeLvl1.UNUSED] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl1.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl1.MISSING] + "</font>)";
			fb.addStaticticsRow("Analysed permission uses:", statsLine);
			if (this.prevResult != null) {
				statsLine = "<strong>"
						+ (prevPermissionUses[ResultTypeLvl1.MAYBE_MISSING]
								+ prevPermissionUses[ResultTypeLvl1.MISSING])
						+ "</strong> (<font color=\"" + ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_MISSING)
						+ "\">" + prevPermissionUses[ResultTypeLvl1.MAYBE_MISSING] + "</font> / <font color=\""
						+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MISSING) + "\">"
						+ prevPermissionUses[ResultTypeLvl1.MISSING] + "</font>) &#8594; ";
			} else {
				statsLine = "";
			}
			statsLine += "<strong>"
					+ (this.permissionUses[ResultTypeLvl1.MAYBE_MISSING] + this.permissionUses[ResultTypeLvl1.MISSING])
					+ "</strong> (<font color=\"" + ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MAYBE_MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl1.MAYBE_MISSING] + "</font> / <font color=\""
					+ ResultTypeLvl1.getColorOfType(ResultTypeLvl1.MISSING) + "\">"
					+ this.permissionUses[ResultTypeLvl1.MISSING] + "</font>)";
			fb.addStaticticsRow("Possibly malicious:", statsLine);
			fb.setAppTrustworthy(this.permissionUsesAll[ResultTypeLvl1.MISSING]
					+ this.permissionUsesAll[ResultTypeLvl1.MAYBE_MISSING] == 0);
			fb.complete();
			bw.close();
		} catch (final IOException e) {
			LOGGER.error("Could not access outputfile.");
		}

		return this.htmlFileTextual.toString();
	}

	private ResultTreeLvl1 getChildren(final Object parent, final DetailLevel inDetailLvl) {
		if (parent == this.app) {
			if (inDetailLvl == DetailLevelLvl1.APP || inDetailLvl == null) {
				return this.classes;
			}
		} else if (parent == this.classes || parent == this.components) {
			if (inDetailLvl == DetailLevelLvl1.APP || inDetailLvl == DetailLevelLvl1.CLASS
					|| inDetailLvl == DetailLevelLvl1.COMPONENT || inDetailLvl == null) {
				return this.methods;
			}
		}
		return null;
	}

	private void getTextualForTree(final ResultTreeLvl1 tree, final DetailLevel inDetailLvl, final HTMLFrameBuilder fb)
			throws IOException {
		getTextualForTree(tree, inDetailLvl, null, fb);
	}

	private void getTextualForTree(final ResultTreeLvl1 tree, final DetailLevel inDetailLvl,
			final ResultLeafLvl1 parentClass, final HTMLFrameBuilder fb) throws IOException {
		if (tree != null) {
			for (final ResultLeafLvl1 childLeaf : tree.getLeafsAsList()) {
				if (tree == this.methods && parentClass != null) {
					if (childLeaf.getName().contains(parentClass.getName())) {
						getTextualForLeaf(childLeaf, getChildren(tree, inDetailLvl), inDetailLvl, fb);
					}
				} else {
					getTextualForLeaf(childLeaf, getChildren(tree, inDetailLvl), inDetailLvl, fb);
				}
			}
		}
	}

	private void getTextualForLeaf(final ResultLeafLvl1 leaf, final ResultTreeLvl1 children,
			final DetailLevel inDetailLvl, final HTMLFrameBuilder fb) throws IOException {
		if (children != null && this.activeFilters.contains(AnalysisResultLvl1.TREEOPTION)) {
			fb.append("<div class=\"plus\" onClick=\"show(this)\"><div class=\"inner\">+</div></div>");
		} else {
			fb.append("<div class=\"spacer\"></div>");
		}
		fb.append(" <strong>");
		if (this.methods.getLeafsAsList().contains(leaf)) {
			fb.append(fixString(leaf.getName()));
		} else {
			fb.append(leaf.getName());
		}
		fb.append("</strong><br />\n" + "<div class=\"premissions\">\n");
		final StringBuilder resultStr = new StringBuilder();
		for (int i = 0; i < 5; i++) {
			if (leaf.getPermissions(i) != null) {
				for (final Permission permission : leaf.getPermissions(i)) {
					resultStr.append(getTextualForPermission(permission, i, leaf));
				}
			}
		}
		fb.append(resultStr.toString());
		fb.append("</div>\n" + "<br />\n");
		if (this.activeFilters.contains(AnalysisResultLvl1.TREEOPTION)) {
			fb.append("<div class=\"popout\">\n" + "\t");
			getTextualForTree(children, inDetailLvl, leaf, fb);
			fb.append("\n</div>\n");
			if (children != null || inDetailLvl == DetailLevelLvl1.METHOD) {
				fb.append("<br />\n");
			}
		}
	}

	private String getTextualForPermission(final Permission permission, final int type, final ResultLeafLvl1 leaf) {
		if (this.activeFilters.contains(ResultTypeLvl1.getStringOfType(type))
				|| this.activeFilters.contains(Constants.PREFIX_PERMISSION + " " + permission.getName())) {
			this.permissionUses[type]++;
			if (this.prevResult == null) {
				return "\t<font color=\"" + ResultTypeLvl1.getColorOfType(type) + "\" title=\""
						+ ResultTypeLvl1.getStringOfType(type) + "\">" + permission.getName() + "</font><br />\n";
			} else {
				final ResultLeafLvl1 prevLeaf = this.mapCurrentToPrev.get(leaf);

				if (prevLeaf != null) {
					int found = -1;
					for (int i = 0; i < 5; i++) {
						if (prevLeaf.getPermissions(i) != null) {
							for (final Permission prevPerm : prevLeaf.getPermissions(i)) {
								if (prevPerm.getName().equals(permission.getName())) {
									found = i;
									break;
								}
							}
							if (found >= 0) {
								break;
							}
						}
					}
					if (type != found) {
						this.changedPermissions++;
						return "\t<strong>" + permission.getName() + "</strong> <font color=\""
								+ ResultTypeLvl1.getColorOfType(found) + "\" title=\""
								+ ResultTypeLvl1.getStringOfType(found) + "\">" + ResultTypeLvl1.getStringOfType(found)
								+ "</font> &#8594; <font color=\"" + ResultTypeLvl1.getColorOfType(type) + "\" title=\""
								+ ResultTypeLvl1.getStringOfType(type) + "\">" + ResultTypeLvl1.getStringOfType(type)
								+ "</font><br />\n";
					} else {
						this.unchangedPermissions++;
						return "\t<font color=\"" + ResultTypeLvl1.getColorOfType(type) + "\" title=\""
								+ ResultTypeLvl1.getStringOfType(type) + "\">" + permission.getName()
								+ "</font> (unchanged)<br />\n";
					}
				} else {
					this.newPermissions++;
					return "\t<font color=\"" + ResultTypeLvl1.getColorOfType(type) + "\" title=\""
							+ ResultTypeLvl1.getStringOfType(type) + "\">" + permission.getName()
							+ "</font> (new)<br />\n";
				}
			}
		} else {
			return "";
		}
	}

	@Override
	public List<DetailLevel> getDetailLevels() {
		final List<DetailLevel> detaillevelList = new ArrayList<>();
		for (final DetailLevelLvl1 item : DetailLevelLvl1.values()) {
			detaillevelList.add(item);
		}
		return detaillevelList;
	}

	@Override
	public List<String> getFilters() {
		return this.filters;
	}

	/**
	 * Replies the map of permissions used in the graphical result.
	 *
	 * @return A map of permissions which maps permissions to graphical objects
	 *         of the graphical result.
	 */
	public Map<Permission, SVGPermissionLvl1> getMapOfPermissions() {
		return this.mapOfPermissions;
	}

	/**
	 * Used to format a string for a better visualization result in the textual
	 * result.
	 *
	 * @param str
	 *            Unformatted string input
	 * @return The formatted string
	 */
	private static String fixString(final String str) {
		return str.substring(1, str.length() - 1).replaceAll("<", "&lt;").replaceAll(">", "&gt;");
	}

	/**
	 * Resets an output file
	 *
	 * @param file
	 *            Defines the file to be reset.
	 * @throws IOException
	 */
	private static void resetFile(final File file) throws IOException {
		file.delete();
		file.createNewFile();
	}
}