/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a.analyzer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.datastructures.Transition;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisGraphLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.ParamType;
import de.upb.pga3.panda2.extension.lvl2a.ParameterNode;
import de.upb.pga3.panda2.extension.lvl2a.TransitionLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.TransitionType;
import soot.Unit;

/**
 * This class perform backward slicing algorithm for a specific sink to get a
 * list of sources
 *
 * @author nptsy
 */
public class BackwardSlicer {

	private static final Logger LOGGER = LogManager.getLogger(BackwardSlicer.class);

	/**
	 * slice the graph based on sink
	 *
	 * @param inGraph
	 *            the graph that is sliced based on sink
	 * @param inSink
	 * @return
	 */
	public Map<Permission, List<Unit>> slice(final AnalysisGraphLvl2a inGraph, final Unit inSink,
			final Map<Permission, List<Unit>> inMapSources) {
		// in case the input graph is null
		if (inGraph == null) {
			LOGGER.error("Cannot find the input graph.");
			return null;
		}

		// in case the slicing criterion is null
		if (inSink == null) {
			LOGGER.error("The slicing criterion is null.");
			return null;
		}

		// in case the inColSources is invalid ==> return null;
		if (inMapSources == null) {
			LOGGER.error("The list of sources is invalid");
			return null;
		}

		// size of found source permission
		final Integer iSizeFoundSources = 0;

		// get size of the map source permissions
		int iSizeMapSources = 0;

		boolean isFullSource = false;

		final Collection<List<Unit>> colLstUnits = inMapSources.values();
		for (final List<Unit> lstUnits : colLstUnits) {
			iSizeMapSources += lstUnits.size();
		}

		// the return map, this map just consists of sources
		final Map<Permission, List<Unit>> mapResults = new HashMap<>();

		// list of already checked objects
		final List<Object> checkedObject = new ArrayList<Object>();

		/*
		 * --------------------------------------------------------------------
		 * process predecessor of unit first
		 * --------------------------------------------------------------------
		 */
		// get set of transitions that have target is the input sink
		final Set<Transition> setTransitions = inGraph.getIncomingTransitions(inSink);

		final List<Object> lstParams = new ArrayList<>();
		/*
		 * preDecessors is a list used for storing predecessors of nodes for
		 * each back ward step traverse
		 */
		// final List<Object> preDecessors =
		// filterPredecessorFirstStep(setTransitions, lstParams);
		final List<Object> preDecessors = filterPredecessor(setTransitions, lstParams, true);

		/*
		 * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
		 * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
		 * the backward slicing for a sink has 2 phases:
		 *
		 * @1st phase: context-sensitive approach => traverse all transitions of
		 * type DataFlow, ControlDependency, CALL, SUMMARY, PARAM_IN
		 *
		 * @2nd phase: starts from PARAMETERNODE and goes backward through
		 * DATAFLOW, CONTROLDEPENDENCY, SUMMARY and PARAMOUT
		 *
		 * The result of 2 phases will be merged
		 * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
		 * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
		 */

		/*
		 * ---------------------------------------------------------------------
		 * 1ST PHASE
		 */

		isFullSource = processGraph(inMapSources, mapResults, inGraph, preDecessors, true, checkedObject,
				iSizeFoundSources, iSizeMapSources, lstParams);

		/*
		 * END FIRST PHASE
		 * ---------------------------------------------------------------------
		 */

		/*
		 * ---------------------------------------------------------------------
		 * 2ND PHASE
		 */
		if (!isFullSource) {
			processGraph(inMapSources, mapResults, inGraph, lstParams, false, checkedObject, iSizeFoundSources,
					iSizeMapSources, lstParams);
		}
		/*
		 * END SECOND PHASE
		 * ---------------------------------------------------------------------
		 */

		return mapResults;
	}

	/**
	 * traverse backward the graph to collect sources
	 *
	 * @param inMapSources
	 *            the map that contains all found sources in the input app
	 * @param mapResults
	 *            the map containing sources which have paths to the specific
	 *            sink
	 * @param inGraph
	 *            the graph (PDG graph)
	 * @param inLstObjs
	 *            the list that contains predecessor
	 * @param isFirstStep
	 *            the first phase or not (the Backward slicing algo has 2
	 *            different phases)
	 * @param checkedObject
	 *            the list of already checked object
	 * @param iSizeFoundSources
	 *            the number of found sources for the sink
	 * @param iSizeMapSources
	 *            the number of sources existing in the app
	 * @return true if number of found sources for the sink is equal to the
	 *         number of sources existing in the app. Otherwise false
	 */
	private boolean processGraph(final Map<Permission, List<Unit>> inMapSources,
			final Map<Permission, List<Unit>> mapResults, final AnalysisGraphLvl2a inGraph,
			final List<Object> inLstObjs, final boolean isFirstStep, final List<Object> checkedObject,
			Integer iSizeFoundSources, final int iSizeMapSources, final List<Object> lstParams) {

		boolean isFullSources = false;

		while (inLstObjs != null && !inLstObjs.isEmpty()) {
			// check the first object in list
			final Object obj1 = inLstObjs.get(0);
			// remove the first object out of list
			inLstObjs.remove(0);

			if (checkedObject.contains(obj1)) {
				/*
				 * in case the current object is already checked ==> then don't
				 * check it again
				 */
				continue;
			} else {
				checkedObject.add(obj1);
			}

			/*
			 * get permission of source (unit) only because the target is
			 * already processed for permission as a sink
			 */

			if (obj1 instanceof Unit) {
				final Unit usource = (Unit) obj1;
				final boolean isFound = processPermission(usource, inMapSources, mapResults);
				if (isFound) {
					iSizeFoundSources++;
				}

				if (iSizeFoundSources == iSizeMapSources) {
					isFullSources = true;
					break;
				}
			}

			collectPredecessor(inGraph, inLstObjs, isFirstStep, obj1, lstParams);
		}

		return isFullSources;
	}

	/**
	 * process a source to get its permission and add that source to the map of
	 * result
	 *
	 * @param usource
	 *            the unit which is considered as a source
	 * @param inMapSources
	 *            the map of sources existing in the app
	 * @param mapResults
	 *            the map that contains the sources for the sink
	 * @return true if a new source a found, otherwise false
	 */
	private boolean processPermission(final Unit usource, final Map<Permission, List<Unit>> inMapSources,
			final Map<Permission, List<Unit>> mapResults) {
		boolean foundNewSource = false;
		// get permission of the unit
		final Set<Permission> setPerms = inMapSources.keySet();
		for (final Permission perm : setPerms) {
			// get unit corresponding to permission
			final List<Unit> lstUnit = inMapSources.get(perm);

			// in case the unit is looked for
			if (lstUnit.contains(usource)) {
				foundNewSource = true;

				List<Unit> lstUnits = mapResults.get(perm);
				if (lstUnits == null) {
					lstUnits = new ArrayList<>();
					lstUnits.add(usource);
					mapResults.put(perm, lstUnits);
				} else {
					lstUnits.add(usource);
				}
			}
		}

		return foundNewSource;
	}

	/**
	 * collect predecessor of an object
	 *
	 * @param inGraph
	 *            the PDG graph
	 * @param inLstObjs
	 *            the list that will contains all predecessors
	 * @param inIsFirstStep
	 *            true if this method processes for first step in the backward
	 *            slicing algo. Otherwise it will be false
	 * @param inCurObj
	 *            the object (node) whose predecessor will be got
	 */
	private void collectPredecessor(final AnalysisGraphLvl2a inGraph, final List<Object> inLstObjs,
			final boolean inIsFirstStep, final Object inCurObj, final List<Object> lstParams) {
		// get transition from the object ob
		final Set<Transition> setTransitions = inGraph.getIncomingTransitions(inCurObj);

		List<Object> tempNextSinks;
		if (!inIsFirstStep) {
			// get predecessor of the setTransitions
			tempNextSinks = filterPredecessor(setTransitions, lstParams, false);
		} else {
			tempNextSinks = filterPredecessor(setTransitions, lstParams, true);
		}

		/*
		 * only add non-existing parent node to the preDecessors list
		 */
		if (tempNextSinks != null && !tempNextSinks.isEmpty()) {
			for (final Object object : tempNextSinks) {
				if (!inLstObjs.contains(object)) {
					final int iLastIndex = inLstObjs.size();
					// preDecessors.addAll(tempNextSinks);
					inLstObjs.add(iLastIndex, object);
				}
			}
		}
	}

	/**
	 * filter predecessor based on 2 phases of backward algo
	 *
	 * @param inSetTransitions
	 *            the set of edges
	 * @param inLstParams
	 *            the list that contains only parameter nodes
	 * @param inIsFirstStep
	 *            flaf for the first phase of second phase
	 * @return a list of object as predecessors
	 */
	private List<Object> filterPredecessor(final Set<Transition> inSetTransitions, final List<Object> inLstParams,
			final boolean inIsFirstStep) {

		// predecessor list
		final List<Object> predecessor = new ArrayList<Object>();

		if (inSetTransitions != null && !inSetTransitions.isEmpty()) {
			for (final Transition objTrans : inSetTransitions) {

				// only check for TransitionLvl2a
				if (objTrans instanceof TransitionLvl2a) {
					final TransitionLvl2a tran2a = (TransitionLvl2a) objTrans;
					final TransitionType tranType = tran2a.getTransitionType();

					// source node
					final Object source = objTrans.getSource();
					final Object target = objTrans.getTarget();

					/*
					 * PHASE 1
					 *
					 * @Note 1 check only for case that the node isn't a
					 * formal_out node
					 *
					 * @Node 2 only check transitions which have type of
					 * CONTROLDEPENDENCY, DATAFLOW, SUMMARY, CALL and PARAMIN.
					 * Because FORMAL_OUT node only exists with transition type
					 * PARAMOUT so we don't need to care it here
					 */
					if (inIsFirstStep) {
						if (tranType == TransitionType.CONTROLDEPENDENCY || tranType == TransitionType.DATAFLOW
								|| tranType == TransitionType.SUMMARY || tranType == TransitionType.CALL
								|| tranType == TransitionType.PARAMIN) {

							/*
							 * @Question check again whether the target is the
							 * input target node or not by using object [target]
							 * ==> NO
							 */
							if (!predecessor.contains(source)) {
								predecessor.add(source);
							}
						} // finish if tranType
						else {
							if (source instanceof ParameterNode) {
								final ParameterNode param = (ParameterNode) source;
								if (param.getType().equals(ParamType.FORMAL_OUT)) {
									inLstParams.add(param);
								}
							}
							if (target instanceof ParameterNode) {
								final ParameterNode param = (ParameterNode) target;
								if (param.getType().equals(ParamType.FORMAL_OUT)) {
									inLstParams.add(param);
								}
							}
						}
					} else {
						if (tranType == TransitionType.CONTROLDEPENDENCY || tranType == TransitionType.DATAFLOW
								|| tranType == TransitionType.SUMMARY || tranType == TransitionType.PARAMOUT) {

							/*
							 * @Question check again whether the target is the
							 * input target node or not by using object [target]
							 * ==> NO
							 */
							if (!predecessor.contains(source)) {
								predecessor.add(source);
							}
						} // finish if tranType
					}
				}
			}
		}
		return predecessor;
	}
}
