package de.upb.pga3.panda2.extension.lvl2a.flowpath;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * Represents a path of (information) flow from a source permission to a sink
 * permission
 *
 * @author Fabian
 *
 */
public class FlowPath implements Iterable<PathElement> {

	final ArrayList<PathElement> mElements;

	/**
	 * Creates an empty path
	 */
	public FlowPath() {
		this.mElements = new ArrayList<>();
	}

	/**
	 * Creates an empty path with given initial capacity
	 *
	 * @param initCapacity
	 *            The initial capacity
	 */
	public FlowPath(final int initCapacity) {
		this.mElements = new ArrayList<>(initCapacity);
	}

	/**
	 * Adds a {@link PathElement} to the path. The first element to be added has
	 * to be a resource. When a second resource is added to the path, the path
	 * will be considered as complete and cannot be changed anymore.
	 *
	 * @param inTrans
	 */
	public void addElement(final PathElement inPathElem) {

		if (inPathElem != null) {
			if (isComplete()) {
				throw new IllegalArgumentException("The path is already complete and nothing more can be added!");
			} else if (this.mElements.isEmpty() && !(inPathElem instanceof ResourceElement)) {
				throw new IllegalArgumentException("The first element of a path must be of type ResourceElement!");
			}
			this.mElements.add(inPathElem);
		} else {
			throw new IllegalArgumentException("The path element to be added must not be empty!");
		}
	}

	public PathElement getElement(final int i) {

		if (i < 0 || i >= this.mElements.size()) {
			throw new IllegalArgumentException("The given elemt index must match 0 <= i < length!");
		}
		return this.mElements.get(i);
	}

	public void trim() {
		this.mElements.trimToSize();
	}

	/**
	 * Returns the sink permission of that path or null if the path is not
	 * complete.
	 *
	 * @return The sink permission
	 */
	public ResourceElement getSink() {

		if (isComplete()) {
			return (ResourceElement) this.mElements.get(this.mElements.size() - 1);
		} else {
			return null;
		}

	}

	/**
	 * Returns the source permission of that path or null if the path is not
	 * complete.
	 *
	 * @return The source permission
	 */
	public ResourceElement getSource() {

		if (isComplete()) {
			return (ResourceElement) this.mElements.get(0);
		} else {
			return null;
		}
	}

	/**
	 * Returns true if the the path start and ends with a permission; returns
	 * false otherwise.
	 *
	 * @return If the path is complete
	 */
	public boolean isComplete() {

		if (this.mElements.size() >= 2) {
			return (this.mElements.get(0) instanceof ResourceElement)
					&& (this.mElements.get(this.mElements.size() - 1) instanceof ResourceElement);
		} else {
			return false;
		}
	}

	/**
	 * Returns the number of elements in that path.
	 *
	 * @return The length of the path
	 */
	public int getLength() {
		return this.mElements.size();
	}

	/**
	 * Returns a new sub-path copy of the common part of this path and the given
	 * path. The comparison will be done from the sources directed to the sinks.
	 *
	 * @param inOther
	 *            The path to be compared
	 * @return The common prefix
	 */
	public FlowPath getCommonPrefix(final FlowPath inOther) {

		final FlowPath common = new FlowPath();
		final Iterator<PathElement> thisIt = iterator();
		final Iterator<PathElement> otherIt = inOther.iterator();
		PathElement elem;
		while (thisIt.hasNext() && otherIt.hasNext()) {
			elem = thisIt.next();
			if (elem.equals(otherIt.next())) {
				common.addElement(elem);
			} else {
				break;
			}
		}
		return common;
	}

	/**
	 * Returns a new sub-path copy defined by the given indexes.
	 *
	 * @param inFromIdx
	 *            The start index (inclusive)
	 * @param inToIdx
	 *            The end index (exclusive)
	 * @return The sub-path
	 */
	public FlowPath getSubPath(final int inFromIdx, final int inToIdx) {

		if (0 <= inFromIdx && inFromIdx <= inToIdx && inToIdx <= getLength()) {
			final FlowPath subPath = new FlowPath();
			subPath.mElements.addAll(this.mElements.subList(inFromIdx, inToIdx));
			return subPath;
		} else {
			throw new IllegalArgumentException("Given indexes are not matching the pattern: 0 <= from <= to <= length");
		}
	}

	@Override
	public Iterator<PathElement> iterator() {

		final Iterator<PathElement> it = new Iterator<PathElement>() {
			Iterator<PathElement> mListIt = FlowPath.this.mElements.iterator();

			@Override
			public boolean hasNext() {
				return this.mListIt.hasNext();
			}

			@Override
			public PathElement next() {
				return this.mListIt.next();
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException("Removing elements from a path is not allowed");
			}
		};
		return it;
	}

	@Override
	public int hashCode() {

		final int prime = 31;
		int result = 1;
		result = prime * result + ((this.mElements == null) ? 0 : this.mElements.hashCode());
		return result;
	}

	@Override
	public boolean equals(final Object obj) {

		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		final FlowPath other = (FlowPath) obj;
		if (this.mElements == null) {
			if (other.mElements != null) {
				return false;
			}
		} else if (!this.mElements.equals(other.mElements)) {
			return false;
		}
		return true;
	}

	@Override
	public String toString() {

		final StringBuilder sb = new StringBuilder();
		for (int i = 0; i < this.mElements.size(); i++) {
			sb.append(this.mElements.get(i).toFullString());
			if (i < (this.mElements.size() - 1)) {
				sb.append("\n################################\n");
			}
		}
		return sb.toString();
	}

}
