package de.upb.pga3.panda2.extension.lvl2b;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import de.upb.pga3.panda2.core.Analysis;
import de.upb.pga3.panda2.core.AnalysisFactory;
import de.upb.pga3.panda2.core.AnalysisProcedure;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.extension.A3AnalysisProcedure;
import de.upb.pga3.panda2.extension.lvl1.analyzer.AnalyzerLvl1;
import de.upb.pga3.panda2.extension.lvl1.graphgenerator.GraphGeneratorLvl1;
import de.upb.pga3.panda2.extension.lvl2b.analyzer.AnalyzerLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.graphgenerator.GraphGeneratorLvl2b;

/**
 * This is the AnalysisFactory for generating any type of level 2b analysis.
 *
 * @author Felix
 */

public class AnalysisFactoryLvl2b implements AnalysisFactory {
	private File apk;
	private File apkForPrevRes;
	private List<File> nnapks;
	private AnalysisResult prevRes;
	private boolean allmode;

	/**
	 * Constructor for the level 2b factory. Providing the parameters will be
	 * enough to create any type of level 2b analysis.
	 *
	 * @param apk
	 *            Input .apk file belonging to the App that is going to be
	 *            analyzed.
	 *
	 * @param nnapks
	 *            This is a collection of non-native .apk(s) to find out the
	 *
	 * @param prevRes
	 *            In comparison mode this parameter will contain the analysis
	 *            result of a previously executed level 2b analysis.
	 */
	public AnalysisFactoryLvl2b(final File apk, final List<File> nnapks, final AnalysisResultLvl2b prevRes,
			final boolean allmode) {
		this.apk = apk;
		this.nnapks = nnapks;
		this.prevRes = prevRes;
		this.allmode = allmode;
		this.apkForPrevRes = null;
	}

	/**
	 * Constructor for the level 2b factory. Providing the parameters will be
	 * enough to create any type of level 2b analysis.
	 *
	 * @param apk
	 *            Input .apk file belonging to the App that is going to be
	 *            analyzed.
	 *
	 * @param nnapks
	 *            This is a collection of non-native .apk(s) to find out the
	 *
	 * @param prevRes
	 *            .apk file that will be used to compute the previous result
	 *            which is then in turn compared to the analyzed App provided by
	 *            the apk parameter.
	 */
	public AnalysisFactoryLvl2b(final File apk, final List<File> nnapks, final File apkForPrevRes,
			final boolean allmode) {
		this.apk = apk;
		this.nnapks = nnapks;
		this.prevRes = null;
		this.allmode = allmode;
		this.apkForPrevRes = apkForPrevRes;
	}

	/**
	 * This method generates a list of analyses to be executed in order to run a
	 * level 2b analysis.
	 *
	 * @return The created list of analyses to be executed.
	 */
	@Override
	public List<Analysis> createAnalysis() {
		// Create list
		final List<Analysis> analysesList = new ArrayList<>();
		final List<Analysis> analysesLvl1 = new ArrayList<>();

		// Create Level 1 Analyses
		if (this.nnapks != null) {
			for (final File nnapk : this.nnapks) {
				final AnalysisProcedure anaProc = new A3AnalysisProcedure(new GraphGeneratorLvl1(), new AnalyzerLvl1());
				final Analysis ana = new Analysis(nnapk.toPath(), null, anaProc);
				analysesLvl1.add(ana);
			}
		}
		analysesList.addAll(analysesLvl1);

		// Create Level 2b Analysis
		if (this.apkForPrevRes != null) {
			// Create comparison Analysis
			AnalysisProcedure anaProcComp = new A3AnalysisProcedure(new GraphGeneratorLvl1(), new AnalyzerLvl1());
			Analysis anaComp = new Analysis(this.apkForPrevRes.toPath(), null, anaProcComp);
			analysesList.add(anaComp);

			anaProcComp = new A3AnalysisProcedure(new GraphGeneratorLvl2b(), new AnalyzerLvl2b(this.allmode));
			anaComp = new Analysis(null, null, anaProcComp);
			analysesList.add(anaComp);

			analysesList.addAll(analysesLvl1);
		}
		AnalysisProcedure anaProc = new A3AnalysisProcedure(new GraphGeneratorLvl1(), new AnalyzerLvl1());
		Analysis ana = new Analysis(this.apk.toPath(), null, anaProc);
		analysesList.add(ana);

		anaProc = new A3AnalysisProcedure(new GraphGeneratorLvl2b(), new AnalyzerLvl2b(this.allmode));
		ana = new Analysis(null, this.prevRes, anaProc);
		analysesList.add(ana);

		return analysesList;
	}

}
