package de.upb.pga3.panda2.utilities;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.util.List;

import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.sun.management.GarbageCollectionNotificationInfo;

/**
 *
 * @author Fabian
 *
 */
public class MemoryUsageMonitor {

	static final Logger LOGGER = LogManager.getLogger(MemoryUsageMonitor.class);

	static final long MB_TO_BYTE = 1048576;
	static final long SEC_TO_MS = 1000;

	private static final MemoryUsageMonitor INSTANCE = new MemoryUsageMonitor();

	public static MemoryUsageMonitor getInstance() {
		return INSTANCE;
	}

	private boolean running;
	private long threshold;
	private boolean warned;

	private int maxMem;
	private int skipSteps;
	private int logCounter;

	private MemoryUsageMonitor() {

		this.running = false;
		this.threshold = 0;
		this.warned = false;
		this.maxMem = Integer.MIN_VALUE;
		this.skipSteps = 1;
		this.logCounter = 0;
	}

	/**
	 * Sets a threshold for showing a warning when used memory reaches threshold
	 *
	 * @param threshold
	 *            The threshold to additionally show a waring when threshold in
	 *            MB is reached; use 0 to disable warning
	 */
	public synchronized void setWarnThreshold(final int threshold) {
		this.threshold = threshold < 0 ? 0 : threshold * MB_TO_BYTE;
	}

	/**
	 * Sets the number of logs to skip. If set to {@code n}, only every
	 * {@code n}th triggering will be logged.
	 *
	 * @param skipSteps
	 *            The number of steps to skip
	 */
	public synchronized void setSkipSteps(final int skipSteps) {
		this.skipSteps = skipSteps < 1 ? 1 : skipSteps;
	}

	/**
	 * Returns the the maximum amount of used memory that was seen by the memory
	 * usage monitor
	 *
	 * @return The max. amount of memory in MB
	 */
	public int getMaxMemSeen() {
		return this.maxMem;
	}

	/**
	 * Sets up a memory usage monitoring that will be run after each garbage
	 * collection
	 *
	 * @param time
	 *            The time interval to run monitoring in seconds; if the
	 *            interval is <= 0 the memory will be logged after each garbage
	 *            collection
	 */
	public synchronized void setupMonitoring(final int timeInterval) {

		if (!this.running) {
			if (timeInterval <= 0) {
				LOGGER.info("Set up memory usage monitoring after each garbage collection");
				setupMonitoringAfterGC();
			} else {
				LOGGER.info("Set up memory usage monitoring after every {} seconds", timeInterval);
				setupPeriodicMonitoring(timeInterval);
			}
			this.running = true;
		} else {
			LOGGER.warn("Memory usage monitoring already running!");
		}
	}

	/**
	 * Sets up a memory usage monitoring that will be run after each garbage
	 * collection
	 */
	private void setupMonitoringAfterGC() {

		// get all the GarbageCollectorMXBeans - there's one for each heap
		// generation; so probably two - the old generation and young generation
		final List<GarbageCollectorMXBean> gcbeans = java.lang.management.ManagementFactory
				.getGarbageCollectorMXBeans();
		// Install a notifcation handler for each bean
		for (final GarbageCollectorMXBean gcbean : gcbeans) {
			final NotificationEmitter emitter = (NotificationEmitter) gcbean;
			emitter.addNotificationListener(new GCListener(), null, null);
		}
	}

	/**
	 * Sets up a memory usage monitoring that will be run periodically
	 *
	 * @param time
	 *            The time interval to run monitoring in seconds
	 */
	private void setupPeriodicMonitoring(final int time) {

		final Thread t = new Thread(new PeriodicMonitor(time));
		t.setDaemon(true);
		t.start();
	}

	void logMem() {

		if (this.logCounter % this.skipSteps == 0) {
			final MemoryUsage mem = ManagementFactory.getMemoryMXBean().getHeapMemoryUsage();
			final int usedMem = (int) Math.ceil((double) mem.getUsed() / MB_TO_BYTE);
			if (usedMem > this.maxMem) {
				this.maxMem = usedMem;
			}
			LOGGER.debug("Used Heapspace: {}MB", usedMem);
			if (this.threshold > 0) {
				if (usedMem > this.threshold && !this.warned) {
					LOGGER.warn("Memory usage threshold of {}MB reached", (this.threshold / MB_TO_BYTE) + 1);
					this.warned = true;
				} else if (usedMem <= this.threshold && this.warned) {
					this.warned = false;
				}
			}
		}
		this.logCounter++;
	}

	private class PeriodicMonitor implements Runnable {

		private final long time;

		public PeriodicMonitor(final int time) {
			super();
			this.time = time * SEC_TO_MS;
		}

		@Override
		public void run() {

			while (true) {
				logMem();

				try {
					Thread.sleep(this.time);
				} catch (final InterruptedException e) {
					LOGGER.error("Thread sleep interrupted: {}", e.getMessage());
				}
			}
		}
	}

	private class GCListener implements NotificationListener {

		public GCListener() {
			super();
		}

		@Override
		public void handleNotification(final Notification notification, final Object handback) {
			// we only handle GARBAGE_COLLECTION_NOTIFICATION notifications here
			if (notification.getType().equals(GarbageCollectionNotificationInfo.GARBAGE_COLLECTION_NOTIFICATION)) {
				logMem();
			}
		}
	}

}
