package de.upb.pga3.panda2.utilities;

import java.util.Collection;
import java.util.LinkedList;

import gnu.trove.map.hash.TCustomHashMap;
import gnu.trove.procedure.TObjectProcedure;
import gnu.trove.set.hash.TCustomHashSet;
import gnu.trove.strategy.HashingStrategy;

/**
 *
 * @author Fabian
 *
 * @param <K>
 * @param <V>
 */
public class TCustomHashMultimap<K, V> extends TCustomHashMap<K, TCustomHashSet<V>> {

	private final HashingStrategy<? super V> valueStrategy;

	public TCustomHashMultimap(final HashingStrategy<? super K> keyStrategy,
			final HashingStrategy<? super V> valueStrategy) {
		super(keyStrategy);
		this.valueStrategy = valueStrategy;
	}

	/**
	 * Computes the total number of values/entries in the multimap
	 *
	 * @return The total number of values
	 */
	public int totalSize() {

		final TTotalSizeProc p = new TTotalSizeProc();
		super.forEachValue(p);
		return p.totalSize;
	}

	/**
	 * States whether the multimap contains the given value of type {@code V}
	 *
	 * @param o
	 *            The value to be checked
	 * @return {@code true} iff the multimap contains the given value
	 */
	@Override
	public boolean containsValue(final Object o) {

		return !super.forEachValue(new TObjectProcedure<TCustomHashSet<V>>() {

			@Override
			public boolean execute(final TCustomHashSet<V> set) {
				return !set.contains(o);
			}
		});
	}

	/**
	 * States whether the multimap contains a set of values of type
	 * {@link TCustomHashSet} (for an arbitrary key)
	 *
	 * @param o
	 *            The set to be checked
	 * @return {@code true} iff the multimap contains the given set
	 */
	public boolean containsValueSet(final Object o) {
		return super.contains(o);
	}

	/**
	 * Stores an entry in the multimap if both key and value are not already
	 * present
	 *
	 * @param key
	 *            The key to be stored
	 * @param value
	 *            The value for the key
	 * @return {@code true} iff the multimap has been changed
	 */
	public boolean putSingleEntry(final K key, final V value) {

		TCustomHashSet<V> set = super.get(key);
		if (set == null) {
			set = new TCustomHashSet<>(this.valueStrategy);
			final boolean changed = super.put(key, set) == null;
			return changed && set.add(value);
		} else {
			return set.add(value);
		}
	}

	/**
	 * Creates a {@link Collection} containing all values in the multimap
	 *
	 * @return All values
	 */
	public Collection<V> singleValues() {

		final Collection<V> vals = new LinkedList<>();
		super.forEachValue(new TObjectProcedure<TCustomHashSet<V>>() {

			@Override
			public boolean execute(final TCustomHashSet<V> set) {
				vals.addAll(set);
				return true;
			}
		});
		return vals;
	}

	/**
	 * Deletes an entry in the multimap if both key and value are present
	 *
	 * @param key
	 *            The key to be deleted
	 * @param value
	 *            The value to be deleted
	 * @return {@code true} iff the multimap has been changed
	 */
	@Override
	public boolean remove(final Object key, final Object value) {

		final TCustomHashSet<V> set = super.get(key);
		if (set == null) {
			return false;
		} else {
			final boolean removed = set.remove(value);
			if (removed && set.isEmpty()) {
				super.remove(key);
			}
			return removed;
		}
	}

	/**
	 * Performs an operation for each value in the multimap
	 *
	 * @param procedure
	 *            The operation to be performed
	 * @return {@code true} iff all values have be traversed
	 */
	public boolean forEachSingleValue(final TObjectProcedure<V> procedure) {

		return super.forEachValue(new TObjectProcedure<TCustomHashSet<V>>() {

			@Override
			public boolean execute(final TCustomHashSet<V> set) {
				return set.forEach(procedure);
			}
		});
	}

	private class TTotalSizeProc implements TObjectProcedure<TCustomHashSet<V>> {

		public int totalSize;

		public TTotalSizeProc() {
			this.totalSize = 0;
		}

		@Override
		public boolean execute(final TCustomHashSet<V> set) {
			this.totalSize += set.size();
			return true;
		}
	}

}
